# Copyright 2022 by Michiel de Hoon.  All rights reserved.
# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.
"""Tests for Align.bed module."""
import os
import tempfile
import unittest
from io import StringIO

import numpy as np

from Bio import Align
from Bio import SeqIO
from Bio.Align import Alignment
from Bio.Seq import Seq
from Bio.SeqRecord import SeqRecord


class TestAlign_dna_rna(unittest.TestCase):
    # The BED file dna_rna.bed was generated using this command:
    # pslToBed dna_rna.psl dna_rna.bed

    def setUp(self):
        data = {}
        records = SeqIO.parse("Blat/dna.fa", "fasta")
        for record in records:
            name, start_end = record.id.split(":")
            assert name == "chr3"
            start, end = start_end.split("-")
            start = int(start)
            end = int(end)
            sequence = str(record.seq)
            assert len(sequence) == end - start
            data[start] = sequence
        self.dna = Seq(data, length=198295559)  # hg38 chr3
        records = SeqIO.parse("Blat/rna.fa", "fasta")
        self.rna = {record.id: record.seq for record in records}

    def check_alignments(self, alignments):
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 1711))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr3")
        self.assertEqual(alignment.query.id, "NR_046654.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[42530895, 42530958, 42532020,
                           42532095, 42532563, 42532606],
                          [     181,      118,      118,
                                 43,       43,        0]])
                # fmt: on
            )
        )
        alignment.target.seq = self.dna
        alignment.query.seq = self.rna[alignment.query.id]
        self.assertTrue(
            np.array_equal(
                alignment.substitutions,
                # fmt: off
            np.array([[36.,  0.,  0.,  0.,  0.,  0.,  0.,  0.],
                      [ 0., 40.,  0.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  0., 57.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  0.,  0., 42.,  0.,  0.,  0.,  0.],
                      [ 2.,  0.,  0.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  1.,  0.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  0.,  3.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  0.,  0.,  0.,  0.,  0.,  0.,  0.],
                     ])
            )
        )
        self.assertEqual(alignment.substitutions.alphabet, "ACGTacgt")
        self.assertEqual(
            str(alignment),
            """\
chr3       42530895 CGGAAGTACTTCTGGGGGTACATACTCATCGGCTGGGGTATGGTACCAGGGAGGGCTTCC
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
NR_046654       181 CGGAAGTACTTCTGGGGGTACATACTCATCGGCTGGGGTATGGTACCAGGGAGGGCTTCC

chr3       42530955 AGGCTGGGGACAGAGGGGGCAAGGCCTGGAGAACTCCCTAGGGGGAGGGTGCCAACCCAG
                 60 |||---------------------------------------------------------
NR_046654       121 AGG---------------------------------------------------------

chr3       42531015 CTTGCAGTCCTACGTCTTGCTTAGCTGCAGGTCCTGCCTGCAAGGATATCAGCCAAGGGT
                120 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531075 CAAGAAAGTCCTCAAAATGTCTGATCCCAGGACAAGTCCCTCAGGTTGCAGCTGCACCTA
                180 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531135 GGGCTGACCTGTGGGACAGATTTTGTGAACATCTTTCCATTTCCCTTTAGTTCCCGAAAT
                240 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531195 ACAcagggccactgctaatctataaagggcctctgtcacaattagaaagagaatgtccgt
                300 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531255 ctaggtagacacagcccttcaggcatacagcttCACCCCCTCAGTGGAGCATCCCTCCGT
                360 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531315 GGTGAACAACCTATGCAACCAAAGACAGCAGACTGACAACCCACCCTTTtctctctccct
                420 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531375 ccctctccctgcttttctccaaaatctctccctcatgccctctacccctgcttcctgtgc
                480 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531435 cctctctgctctttcactctccctGGGCCTGACAGGGGTACCCAGCACATTCACCATGGT
                540 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531495 GTGGACCATCGCCAGGATCCATTTTGAGGATTATGGGTGAGCTGCTGCCCCACACACTCC
                600 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531555 CCCGGCCGCCATCACTTGGGCAGGCCCCCTGGGTGGGATGATAATGCCATCTGGCCTTGG
                660 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531615 TGAGTGGACAAAAACCACAGCTCTCGGGCCAGAGGGGAGGCTGGAGGAGGACCTGGGGAG
                720 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531675 CAACAGACTCTGGGCCCGGGGTTGCTAAAGTGCTCAGGAGCAGAGCTGGGGACAACTGGG
                780 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531735 GGAGGTGCTGCTGAGTCTCTCTCTGGCTGAGGACAATCCCTCTCATTCCTCCCCACGGTC
                840 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531795 TGCTCAGGTGCTGGGACACCATCAACTCCTCACTGTGGTGGATCATAAAGGGCCCCATCC
                900 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531855 TCACCTCCATCTTGGTAAGATaccctcccaccacctagagatggggaaacaggcccaaag
                960 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531915 ggcaggcaacttagcccaaggtcacatgggaaattagtatctaggtcagaactgaaacgt
               1020 ------------------------------------------------------------
NR_046654       118 ------------------------------------------------------------

chr3       42531975 agcttcctaatgcccaatgcaggatcatccccacccctgtcctaccagTTCTTCCTTGAG
               1080 ---------------------------------------------...||||||||||||
NR_046654       118 ---------------------------------------------CAGTTCTTCCTTGAG

chr3       42532035 CGTAAGCGGATTGGGAGCACAGTCCTTAGGGATTTGAAGGAGGTAGAGTTCCCGGATGAC
               1140 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
NR_046654       103 CGTAAGCGGATTGGGAGCACAGTCCTTAGGGATTTGAAGGAGGTAGAGTTCCCGGATGAC

chr3       42532095 CTGCCCAAAGGGGAAATGCCAGAGGAGAGGTAAGATAGAGAGAGGGGCAGCAGGACCCTG
               1200 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532155 GGAAAGAAGACAGGCCAGCAGTCAAGGGGCCTGAACACCTCAGCCTTCCCGCTCTGACTG
               1260 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532215 CCCGAACTCGGGTCCCCACCCACTAGGTAAACTTCATCCTGTTTATTTGCATCATCCGAA
               1320 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532275 TCCTGCTTCAGAAACTGCGGCCCCCAGATATCAGGAAGAGTGACAGCAGTCCATACTCGT
               1380 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532335 GAGTGTGGGCCTAGTGCCTCAGCCCCCAGTACCTCCATCCCCAGTCCTCAAATCATCCCA
               1440 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532395 CATCTCCTTGAAGTCCTCCCACCCCAAACATCCAGAGTCACCAAAGAGCCACATTGTTCT
               1500 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532455 TTCCCACCTCCACCATGGCCTGGCTcagcccaccaccatcccctgctccagccccaccct
               1560 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532515 caCCAGGCTGCACTCAGAGCCCTGCATGCTTCTCCTGCCCACACTCACCTAGCATCCTTC
               1620 ------------------------------------------------||||||||||||
NR_046654        43 ------------------------------------------------CTAGCATCCTTC

chr3       42532575 CCAGGTATGCATCTGCTGCCAAGCCAGGgag 42532606
               1680 ||||||||||||||||||||||||||||...     1711
NR_046654        31 CCAGGTATGCATCTGCTGCCAAGCCAGGGAG        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr3	42530895	42532606	NR_046654.1	1000	-	42530895	42532606	0	3	63,75,43,	0,1125,1668,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (181 aligned letters; 175 identities; 6 mismatches; 1530 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 181:
        identities = 175,
        mismatches = 6.
    gaps = 1530:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 1530:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 1530:
                open_internal_deletions = 2,
                extend_internal_deletions = 1528;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 1530)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 1530)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 1530)
        self.assertEqual(counts.gaps, 1530)
        self.assertEqual(counts.aligned, 181)
        self.assertEqual(counts.identities, 175)
        self.assertEqual(counts.mismatches, 6)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 978)
        self.assertEqual(alignment.shape, (2, 1711))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr3")
        self.assertEqual(alignment.query.id, "NR_046654.1_modified")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[42530895, 42530922, 42530958, 42532020, 42532037,
                           42532039, 42532095, 42532563, 42532606],
                          [     179,      152,      116,      116,       99,
                                 99,       43,       43,        0]])
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr3       42530895 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
NR_046654       179 ????????????????????????????????????????????????????????????

chr3       42530955 ????????????????????????????????????????????????????????????
                 60 |||---------------------------------------------------------
NR_046654       119 ???---------------------------------------------------------

chr3       42531015 ????????????????????????????????????????????????????????????
                120 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531075 ????????????????????????????????????????????????????????????
                180 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531135 ????????????????????????????????????????????????????????????
                240 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531195 ????????????????????????????????????????????????????????????
                300 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531255 ????????????????????????????????????????????????????????????
                360 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531315 ????????????????????????????????????????????????????????????
                420 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531375 ????????????????????????????????????????????????????????????
                480 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531435 ????????????????????????????????????????????????????????????
                540 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531495 ????????????????????????????????????????????????????????????
                600 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531555 ????????????????????????????????????????????????????????????
                660 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531615 ????????????????????????????????????????????????????????????
                720 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531675 ????????????????????????????????????????????????????????????
                780 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531735 ????????????????????????????????????????????????????????????
                840 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531795 ????????????????????????????????????????????????????????????
                900 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531855 ????????????????????????????????????????????????????????????
                960 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531915 ????????????????????????????????????????????????????????????
               1020 ------------------------------------------------------------
NR_046654       116 ------------------------------------------------------------

chr3       42531975 ????????????????????????????????????????????????????????????
               1080 ---------------------------------------------|||||||||||||||
NR_046654       116 ---------------------------------------------???????????????

chr3       42532035 ????????????????????????????????????????????????????????????
               1140 ||--||||||||||||||||||||||||||||||||||||||||||||||||||||||||
NR_046654       101 ??--????????????????????????????????????????????????????????

chr3       42532095 ????????????????????????????????????????????????????????????
               1200 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532155 ????????????????????????????????????????????????????????????
               1260 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532215 ????????????????????????????????????????????????????????????
               1320 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532275 ????????????????????????????????????????????????????????????
               1380 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532335 ????????????????????????????????????????????????????????????
               1440 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532395 ????????????????????????????????????????????????????????????
               1500 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532455 ????????????????????????????????????????????????????????????
               1560 ------------------------------------------------------------
NR_046654        43 ------------------------------------------------------------

chr3       42532515 ????????????????????????????????????????????????????????????
               1620 ------------------------------------------------||||||||||||
NR_046654        43 ------------------------------------------------????????????

chr3       42532575 ??????????????????????????????? 42532606
               1680 |||||||||||||||||||||||||||||||     1711
NR_046654        31 ???????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr3	42530895	42532606	NR_046654.1_modified	978	-	42530895	42532606	0	5	27,36,17,56,43,	0,27,1125,1144,1668,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (179 aligned letters; 0 identities; 0 mismatches; 1532 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 179:
        identities = 0,
        mismatches = 0.
    gaps = 1532:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 1532:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 1532:
                open_internal_deletions = 3,
                extend_internal_deletions = 1529;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 1532)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 1532)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 1532)
        self.assertEqual(counts.gaps, 1532)
        self.assertEqual(counts.aligned, 179)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 5407))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr3")
        self.assertEqual(alignment.query.id, "NR_111921.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[48663767, 48663813, 48665640,
                           48665722, 48669098, 48669174],
                          [       0,       46,       46,
                                128,      128,      204]])
                # fmt: on
            )
        )
        alignment.target.seq = self.dna
        alignment.query.seq = self.rna[alignment.query.id]
        self.assertTrue(
            np.array_equal(
                alignment.substitutions,
                # fmt: off
            np.array([[53.,  0.,  0.,  0.,  0.,  0.,  0.,  0.],
                      [ 0., 35.,  0.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  0., 50.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  0.,  0., 27.,  0.,  0.,  0.,  0.],
                      [ 9.,  0.,  0.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  7.,  0.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  0., 16.,  0.,  0.,  0.,  0.,  0.],
                      [ 0.,  0.,  0.,  7.,  0.,  0.,  0.,  0.],
                     ])
            )
        )
        self.assertEqual(alignment.substitutions.alphabet, "ACGTacgt")
        self.assertEqual(
            str(alignment),
            """\
chr3       48663767 CACGAGAGGAGCGGAGGCGAGGGGTGAACGCGGAGCACTCCAATCGGTCAGTCATTGTTT
                  0 ||||||||||||||||||||||||||||||||||||||||||||||--------------
NR_111921         0 CACGAGAGGAGCGGAGGCGAGGGGTGAACGCGGAGCACTCCAATCG--------------

chr3       48663827 CTATTGGCACAATGGGAGGCCCCGCCCCTCACGGCGGACTCATCGCATGGGGGAGGGGGC
                 60 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48663887 TCCGCGGGTTGCCGGCTAACCGTGAGAGAGTCCGGGAGGTACACTATACGGACCGGCCTC
                120 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48663947 CAAAGGCGGAATCGATAACGAGCTGCAGCGCCGGGTGCAGAGGACGCGGGCATCCCGAAG
                180 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664007 CCCAGGAAGAGGTCAGGGCCGGGACCCCAGAACGCTCCACAGGGTGCGGCTCCCGCGATG
                240 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664067 GGGTGGATCCTGGTTCTAACAGGCGAGGAACTCCTGGCCAAGGCCTCTGGCCCGCCCCGA
                300 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664127 ACGGTCCCTATGACATCACCATCAACCAATCAGTCGGCGCATCCTTTCGCCCCTTGACTG
                360 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664187 CTCCGCTTCCGGGAGGCGGGGCTTCTGCGGGTTCCACCTCCCGAGCGCCCCTTGTGGCTA
                420 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664247 CCAAGGTCAGGCAACAGGTGTCCAGTTGTTCCCTCTCCTGTCTACGAATCTGAGGACCTC
                480 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664307 CCCAGGATCAGAGCTCTGGGCCTGATACACGGCCGGGGTTCCTACGGGTTTGTGAGTGGG
                540 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664367 GGTGGAAGATCTGCAGAGGCACTTAGGGCTGAACTCCTTTGAATGGGAGCCAATCGGTGC
                600 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664427 AGGGGCTGGAGGAGCGAGTCCCCCAAAGTAGttttatttatctatttagagacaaggtct
                660 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664487 cactctttcggagtgcagtggtgatcacagctcaccgtagcctcgaactccccaggcgat
                720 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664547 tctctcacctcagcctcccgagtagctgggactacgggtacatgtcatcacacttggcta
                780 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664607 atttttgcattttttatagagacagggtctcaccatgtaggccagattagtcttgaactc
                840 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664667 ctgggctcaagcaatccgcccatcttggcctcccaaagtgctgggattataggtgtgagc
                900 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664727 caccgcgcccggcAACCCAGAAGTGGTTTTGACAGCAccagcgctttctgtgtccacaat
                960 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664787 ctagtgagtagagggcacaaaacctgacaccacggaggcagacaggcaggggctctgccg
               1020 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664847 gggaagggtgttggagtcccaaaggaggcgtctgagtcaccttcgcaacctgggacgcct
               1080 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664907 tcttgcataagatgcctgagcagtgccttgaatgaccaaggggagatccgcatctgcaaa
               1140 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48664967 ggaagggcagggagggatagggattgggggtgggcatcctaggtcttggagactgtgtgg
               1200 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665027 gcaaatgtgcagagacataaagggactatggctgagggaaatcaagCCCTGCCCTCTCAC
               1260 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665087 CAATAGGGCTGGCGCTGGTCCCAGCTAACACTCCTTTTGGAGAGCAAAGCTCCTCACTTC
               1320 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665147 TGAGTAGTGAGATTGATTGCGGATCACTCTCCATGTTGCTGCCTGCTGTGTGTCATCCCA
               1380 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665207 CTGTCATCCTCCCTTTGTGGCTGTTCTGTGGAGCCCCTCTCCCTCAATCTGCACTCACCT
               1440 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665267 CTATGCCCCAGCCCCATTGGCAGCTCCTAATGCACTCCCGGTaaaaaaaaaaaaacaaaa
               1500 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665327 aCCAGATGTTAGTGATAGTGGTGGTAGTTCTTCTCTCCACCTCCAAATCTTGCCCTTGCC
               1560 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665387 TCCTAATAAGACCCCTATGTGGTTTAACCTCAttttttttttttttttttttttttgaga
               1620 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665447 tggagtttcactctgtcacccaggctggagtgaagtggtgtgatGGGGCTTCACCATGTg
               1680 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665507 atggggcttcaccatgttggccaggctggtatcaaactcctgacctctagtgatctgccc
               1740 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665567 gcctcagcctcccaaagtgctgggattaccggcatgaggcaccgtgcccagccTATCCTC
               1800 ------------------------------------------------------------
NR_111921        46 ------------------------------------------------------------

chr3       48665627 CTTCTCTTATCAGCTCCCAACTAGAGGTCCACCCAGGACCCAGAGACCTGGATTTGAGGC
               1860 -------------|||||||||||||||||||||||||||||||||||||||||||||||
NR_111921        46 -------------CTCCCAACTAGAGGTCCACCCAGGACCCAGAGACCTGGATTTGAGGC

chr3       48665687 TGCTGGGCGGCAGATGGAGCGATCAGAAGACCAGGGTAAGGGTGTGGCAGATACTGCCAC
               1920 |||||||||||||||||||||||||||||||||||-------------------------
NR_111921        93 TGCTGGGCGGCAGATGGAGCGATCAGAAGACCAGG-------------------------

chr3       48665747 TAACACTTCTCAGCCTTTCCTTCTCCTGCCTTTTCCACCCCACCCTGTGTTTGTCTACTC
               1980 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48665807 CCAGCCAGGTGTACCTTTCCAGGGGAAGACCTGGCCAACCTGTCCAGCTCAATTAtccag
               2040 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48665867 cagttctttgacctcactgagatctcgagtccattgttcatcacctcagctattgacctg
               2100 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48665927 tgtcattagccttatagagttcagtgccacggaaactccctgccctgttctttttctttt
               2160 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48665987 tctttttttttttttttttgagacagagccttgctctgtcgcccaggctggagtgcagtg
               2220 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666047 gcgcgatctcggctcactgcaagctctgcctcccaggttcacaccattctcctgactcag
               2280 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666107 cctcccgagtagctgggactacaggcgtccaccaccatgcccagctaatttttttttttg
               2340 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666167 tatttttagtagagacggcgtttcaccgtgttagccaggctggtctcgatctcctgacct
               2400 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666227 tgtgatgctcccgcctcggcctcccaaagtgctgggattacaggcatgagccattgtgcc
               2460 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666287 cggcctgccctgttcttcttagacaaacttgctgggctaaaatctaaccccgttaaaata
               2520 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666347 gactatttacgtattgtttgcctctagcgcagcagaacattgctggagaaaaacaaacaa
               2580 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666407 ccgtgctaattggtctcattttatattcatgaccacaagcctcagtattatatcggaggg
               2640 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666467 cctatccagtgcagtagggcaagaaaaataataagttatgaagattggaagggaaaaaaa
               2700 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666527 actaattcacaagcagtaggattgtatatgtaaaaatttcaaaggaacctataggtaagt
               2760 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666587 tgttagaatgagttcagcaaagttgttggacacaagatcaatatataaaaatcagttgca
               2820 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666647 atttctatatgtcaccaacagttagaaaataaatttcttgcctgggcatgttggctcaag
               2880 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666707 cctgtaatcccagcactttgggtggccaaggcgggcagatcacctgaggtcaggagtttg
               2940 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666767 agaccagtttggccaacatggtgaaatcccgtctctactaaaaatacagaaattagccgg
               3000 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666827 gcgtggtggtgggcacctgtagtcccagctactgaggaggctgaggcaggagaatcactt
               3060 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666887 gaacctgggaggcagaggttgcagtgaacgagaaaaaaaaattttttttcttaaaaacaa
               3120 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48666947 tgatgtttacaatagcatcaagtaatatcaaatgctgaggaataaacctaatgaaagatg
               3180 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667007 tgcaaagactacatacacacatacaaaaaaactataaaacattattgagggaaataaaga
               3240 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667067 cataggcctggcattggtggctcatgcctgaaatctcagcactttggagggccaaggtgg
               3300 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667127 gtggatcatttgaggtcaggagttagagatcagtccggccaacatggtgaaacctcatct
               3360 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667187 ctactaaaaatacaaaaaaattagcttggccaggtgcagtggctcacacctgtaatccca
               3420 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667247 gcactttgggaggctgaggcgggcggatcatgaggtcaggagatcgagaccatcctggct
               3480 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667307 aacacggtgaaaccctgtctctactaaaaatacaaaaaaaaattagccgggcctgatggc
               3540 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667367 gggcgcccgtagtcccagctactcgggaggctgaggtagcagaatggcgtgaacctggga
               3600 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667427 ggtgcagcttgcagtgagcctaaattgcgccactgcactccagcctgggtaacagagcga
               3660 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667487 gactccgtttcaaaaaaaaaaaaaaaaattagctgggcatgctgttgtgcacctgcaatc
               3720 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667547 ccagctactctggaggatgaggcagaagtgcctgaacctgggacacagaggttgcagtga
               3780 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667607 gccaagatcatgccattgcactccagcctggacaacacagccagacgctatctgaaaaaa
               3840 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667667 aaaaaaaaaaaaaaagtaaaaaaaatgagaaataaagacataaataaagtgaaaaattgt
               3900 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667727 tccaatattggaaaagtcaatattataaaggtgccaattttcccaaattgatatatggat
               3960 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667787 tcgatgcaacttcagttaaaaatcccactaaattttggctgggtgcggtggctcacacct
               4020 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667847 gtaatcccagcactttgggaggctgaggcgggcggatcacaaggtcaggagatcgagacc
               4080 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667907 atcttggctaacatggtgaaaccgtctctactaaaaatacaaaagttagccgggtgtggt
               4140 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48667967 ggcgggcacctgtagtcccagctacttgggaggctgagacagaatggcgtgaacctgggg
               4200 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668027 aggcggagcttgcagtgagccaagttgacgccactgcactccagcctgggcgacagagca
               4260 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668087 agactctgtctcaaaaaaaaaaaaaaaaaaaTCCCACTAGATTTTGTGTGTGTGTAAACT
               4320 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668147 GACAAACTAGATTTAGcagcctgagcaacacagcaaaaccccatctctacaaaaaataca
               4380 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668207 aaaattttgcacatgcctgtatagtcccagctacttgggaggctgaagtgggaggatcat
               4440 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668267 gtgagctctggggaggtcgaggctgtagtgagctatgatcacatgctgcactctagcctg
               4500 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668327 ggcaacagagcaagagaccctgtatctaaaaaaagaatgaaaattaaaaaataaaaaGAa
               4560 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668387 accaagattgtgtggtactggtacgaggataggaagactaaaggaacgaaatccagagac
               4620 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668447 aggcctgaagatgtgtggaaacttgaattttgacaagggtgGTTCTTCAGAGCTAACATG
               4680 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668507 AAGAAAGGGTTGTTTTCTTTTTTTTGTTTCCCcaggagcaactctattaactgaaagaat
               4740 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668567 aggcttttcaataaatgatgctgggtcagttggatatccatatagaaaaaattaaatgag
               4800 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668627 atctctatttcacactgcttgcataatcaattccatataaatttgacatctgaaaatata
               4860 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668687 cagtttctagaaaacagtatTAAGACCttgttttgttttttgttgttgttgttttttgtt
               4920 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668747 ttgttttttgttttttgagacagagtctcgctctgtcgccaggctggaatacagtggtgc
               4980 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668807 aaccttggctcactgcaacctctgactccctagttcaagcaattctcctgcctcagcctc
               5040 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668867 ccgagtagctgcgattacaggcacatgccaccacgcccagctaatttttgtatttttagt
               5100 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668927 agagatgggggtttcaccatgttggccaggatggtctcgatctcctgaccctgtaatccg
               5160 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48668987 cccacctcggcctcccaaagtgctgggattacaggcgtgagccactgcacctggccAAGA
               5220 ------------------------------------------------------------
NR_111921       128 ------------------------------------------------------------

chr3       48669047 GAAGATCTTAAAGGTGACTTTAAGCAAACttttttttttttttttttacagagacgggag
               5280 ---------------------------------------------------.........
NR_111921       128 ---------------------------------------------------AGACGGGAG

chr3       48669107 ctggagtgcagtggctgttcacaagcgtgaAAGCAAAGATTAAAAAATTTGTTTTTATAT
               5340 ..............................||||||||||||||||||||||||||||||
NR_111921       137 CTGGAGTGCAGTGGCTGTTCACAAGCGTGAAAGCAAAGATTAAAAAATTTGTTTTTATAT

chr3       48669167 TAAAAAA 48669174
               5400 |||||||     5407
NR_111921       197 TAAAAAA      204
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr3	48663767	48669174	NR_111921.1	1000	+	48663767	48669174	0	3	46,82,76,	0,1873,5331,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (204 aligned letters; 165 identities; 39 mismatches; 5203 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 204:
        identities = 165,
        mismatches = 39.
    gaps = 5203:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 5203:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 5203:
                open_internal_deletions = 2,
                extend_internal_deletions = 5201;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 5203)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 5203)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 5203)
        self.assertEqual(counts.gaps, 5203)
        self.assertEqual(counts.aligned, 204)
        self.assertEqual(counts.identities, 165)
        self.assertEqual(counts.mismatches, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 972)
        self.assertEqual(alignment.shape, (2, 5407))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr3")
        self.assertEqual(alignment.query.id, "NR_111921.1_modified")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[48663767, 48663795, 48663796, 48663813, 48665640,
                           48665716, 48665722, 48669098, 48669174],
                          [       0,       28,       28,       45,       45,
                                121,      127,      127,      203]])
                # fmt: on
            )
        )
        # The modified RNAs have gaps in their sequence. As this information is
        # not stored in a BED file, we cannot calculate the substitution matrix.
        self.assertEqual(
            str(alignment),
            """\
chr3       48663767 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||-|||||||||||||||||--------------
NR_111921         0 ????????????????????????????-?????????????????--------------

chr3       48663827 ????????????????????????????????????????????????????????????
                 60 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48663887 ????????????????????????????????????????????????????????????
                120 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48663947 ????????????????????????????????????????????????????????????
                180 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664007 ????????????????????????????????????????????????????????????
                240 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664067 ????????????????????????????????????????????????????????????
                300 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664127 ????????????????????????????????????????????????????????????
                360 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664187 ????????????????????????????????????????????????????????????
                420 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664247 ????????????????????????????????????????????????????????????
                480 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664307 ????????????????????????????????????????????????????????????
                540 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664367 ????????????????????????????????????????????????????????????
                600 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664427 ????????????????????????????????????????????????????????????
                660 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664487 ????????????????????????????????????????????????????????????
                720 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664547 ????????????????????????????????????????????????????????????
                780 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664607 ????????????????????????????????????????????????????????????
                840 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664667 ????????????????????????????????????????????????????????????
                900 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664727 ????????????????????????????????????????????????????????????
                960 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664787 ????????????????????????????????????????????????????????????
               1020 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664847 ????????????????????????????????????????????????????????????
               1080 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664907 ????????????????????????????????????????????????????????????
               1140 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48664967 ????????????????????????????????????????????????????????????
               1200 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665027 ????????????????????????????????????????????????????????????
               1260 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665087 ????????????????????????????????????????????????????????????
               1320 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665147 ????????????????????????????????????????????????????????????
               1380 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665207 ????????????????????????????????????????????????????????????
               1440 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665267 ????????????????????????????????????????????????????????????
               1500 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665327 ????????????????????????????????????????????????????????????
               1560 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665387 ????????????????????????????????????????????????????????????
               1620 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665447 ????????????????????????????????????????????????????????????
               1680 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665507 ????????????????????????????????????????????????????????????
               1740 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665567 ????????????????????????????????????????????????????????????
               1800 ------------------------------------------------------------
NR_111921        45 ------------------------------------------------------------

chr3       48665627 ????????????????????????????????????????????????????????????
               1860 -------------|||||||||||||||||||||||||||||||||||||||||||||||
NR_111921        45 -------------???????????????????????????????????????????????

chr3       48665687 ????????????????????????????????????????????????????????????
               1920 |||||||||||||||||||||||||||||||||||-------------------------
NR_111921        92 ???????????????????????????????????-------------------------

chr3       48665747 ????????????????????????????????????????????????????????????
               1980 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48665807 ????????????????????????????????????????????????????????????
               2040 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48665867 ????????????????????????????????????????????????????????????
               2100 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48665927 ????????????????????????????????????????????????????????????
               2160 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48665987 ????????????????????????????????????????????????????????????
               2220 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666047 ????????????????????????????????????????????????????????????
               2280 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666107 ????????????????????????????????????????????????????????????
               2340 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666167 ????????????????????????????????????????????????????????????
               2400 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666227 ????????????????????????????????????????????????????????????
               2460 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666287 ????????????????????????????????????????????????????????????
               2520 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666347 ????????????????????????????????????????????????????????????
               2580 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666407 ????????????????????????????????????????????????????????????
               2640 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666467 ????????????????????????????????????????????????????????????
               2700 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666527 ????????????????????????????????????????????????????????????
               2760 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666587 ????????????????????????????????????????????????????????????
               2820 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666647 ????????????????????????????????????????????????????????????
               2880 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666707 ????????????????????????????????????????????????????????????
               2940 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666767 ????????????????????????????????????????????????????????????
               3000 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666827 ????????????????????????????????????????????????????????????
               3060 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666887 ????????????????????????????????????????????????????????????
               3120 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48666947 ????????????????????????????????????????????????????????????
               3180 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667007 ????????????????????????????????????????????????????????????
               3240 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667067 ????????????????????????????????????????????????????????????
               3300 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667127 ????????????????????????????????????????????????????????????
               3360 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667187 ????????????????????????????????????????????????????????????
               3420 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667247 ????????????????????????????????????????????????????????????
               3480 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667307 ????????????????????????????????????????????????????????????
               3540 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667367 ????????????????????????????????????????????????????????????
               3600 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667427 ????????????????????????????????????????????????????????????
               3660 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667487 ????????????????????????????????????????????????????????????
               3720 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667547 ????????????????????????????????????????????????????????????
               3780 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667607 ????????????????????????????????????????????????????????????
               3840 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667667 ????????????????????????????????????????????????????????????
               3900 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667727 ????????????????????????????????????????????????????????????
               3960 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667787 ????????????????????????????????????????????????????????????
               4020 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667847 ????????????????????????????????????????????????????????????
               4080 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667907 ????????????????????????????????????????????????????????????
               4140 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48667967 ????????????????????????????????????????????????????????????
               4200 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668027 ????????????????????????????????????????????????????????????
               4260 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668087 ????????????????????????????????????????????????????????????
               4320 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668147 ????????????????????????????????????????????????????????????
               4380 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668207 ????????????????????????????????????????????????????????????
               4440 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668267 ????????????????????????????????????????????????????????????
               4500 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668327 ????????????????????????????????????????????????????????????
               4560 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668387 ????????????????????????????????????????????????????????????
               4620 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668447 ????????????????????????????????????????????????????????????
               4680 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668507 ????????????????????????????????????????????????????????????
               4740 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668567 ????????????????????????????????????????????????????????????
               4800 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668627 ????????????????????????????????????????????????????????????
               4860 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668687 ????????????????????????????????????????????????????????????
               4920 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668747 ????????????????????????????????????????????????????????????
               4980 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668807 ????????????????????????????????????????????????????????????
               5040 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668867 ????????????????????????????????????????????????????????????
               5100 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668927 ????????????????????????????????????????????????????????????
               5160 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48668987 ????????????????????????????????????????????????????????????
               5220 ------------------------------------------------------------
NR_111921       127 ------------------------------------------------------------

chr3       48669047 ????????????????????????????????????????????????????????????
               5280 ---------------------------------------------------|||||||||
NR_111921       127 ---------------------------------------------------?????????

chr3       48669107 ????????????????????????????????????????????????????????????
               5340 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
NR_111921       136 ????????????????????????????????????????????????????????????

chr3       48669167 ??????? 48669174
               5400 |||||||     5407
NR_111921       196 ???????      203
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr3	48663767	48669174	NR_111921.1_modified	972	+	48663767	48669174	0	5	28,17,76,6,76,	0,29,1873,1949,5331,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (203 aligned letters; 0 identities; 0 mismatches; 5204 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 203:
        identities = 0,
        mismatches = 0.
    gaps = 5204:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 5204:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 5204:
                open_internal_deletions = 3,
                extend_internal_deletions = 5201;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 5204)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 5204)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 5204)
        self.assertEqual(counts.gaps, 5204)
        self.assertEqual(counts.aligned, 203)
        self.assertRaises(StopIteration, next, alignments)

    def test_reading(self):
        """Test parsing dna_rna.bed."""
        path = "Blat/dna_rna.bed"
        alignments = Align.parse(path, "bed")
        self.check_alignments(alignments)
        alignments = iter(alignments)
        self.check_alignments(alignments)
        with Align.parse(path, "bed") as alignments:
            self.check_alignments(alignments)
        with self.assertRaises(AttributeError):
            alignments._stream
        with Align.parse(path, "bed") as alignments:
            pass
        with self.assertRaises(AttributeError):
            alignments._stream

    def test_writing(self):
        """Test writing the alignments in dna_rna.bed."""
        path = "Blat/dna_rna.bed"
        with open(path) as stream:
            original_data = stream.read()
        alignments = Align.parse(path, "bed")
        stream = StringIO()
        n = Align.write(alignments, stream, "bed")
        self.assertEqual(n, 4)
        stream.seek(0)
        written_data = stream.read()
        stream.close()
        self.assertEqual(original_data, written_data)


class TestAlign_dna(unittest.TestCase):
    def test_reading_psl_34_001(self):
        """Test parsing psl_34_001.bed."""
        path = "Blat/psl_34_001.bed"
        alignments = Align.parse(path, "bed")
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 16))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr4")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[61646095, 61646111],
                          [       0,       16]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr4       61646095 ???????????????? 61646111
                  0 ||||||||||||||||       16
hg19_dna          0 ????????????????       16
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr4	61646095	61646111	hg19_dna	1000	+	61646095	61646111	0	1	16,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (16 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 16:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 16)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 33))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[10271783, 10271816],
                          [       0,       33]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       10271783 ????????????????????????????????? 10271816
                  0 |||||||||||||||||||||||||||||||||       33
hg19_dna          0 ?????????????????????????????????       33
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	10271783	10271816	hg19_dna	1000	+	10271783	10271816	0	1	33,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (33 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 33:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 33)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 17))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[53575980, 53575997],
                          [      17,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2       53575980 ????????????????? 53575997
                  0 |||||||||||||||||       17
hg19_dna         17 ?????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	53575980	53575997	hg19_dna	1000	-	53575980	53575997	0	1	17,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (17 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 17:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 17)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 854)
        self.assertEqual(alignment.shape, (2, 41))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr9")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[85737865, 85737906],
                          [       0,       41]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr9       85737865 ????????????????????????????????????????? 85737906
                  0 |||||||||||||||||||||||||||||||||||||||||       41
hg19_dna          0 ?????????????????????????????????????????       41
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr9	85737865	85737906	hg19_dna	854	+	85737865	85737906	0	1	41,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (41 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 41:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 41)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 41))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr8")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[95160479, 95160520],
                          [       0,       41]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr8       95160479 ????????????????????????????????????????? 95160520
                  0 |||||||||||||||||||||||||||||||||||||||||       41
hg19_dna          0 ?????????????????????????????????????????       41
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr8	95160479	95160520	hg19_dna	1000	+	95160479	95160520	0	1	41,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (41 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 41:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 41)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 834)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr22")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[42144400, 42144436],
                          [       0,       36]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr22      42144400 ???????????????????????????????????? 42144436
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna          0 ????????????????????????????????????       36
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr22	42144400	42144436	hg19_dna	834	+	42144400	42144436	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 682)
        self.assertEqual(alignment.shape, (2, 44))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[183925984, 183925990, 183926028],
                          [        0,         6,        44]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2      183925984 ???????????????????????????????????????????? 183926028
                  0 ||||||||||||||||||||||||||||||||||||||||||||        44
hg19_dna          0 ????????????????????????????????????????????        44
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	183925984	183926028	hg19_dna	682	+	183925984	183926028	0	2	6,38,	0,6,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (44 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 44:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 44)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 890)
        self.assertEqual(alignment.shape, (2, 170))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[35483340, 35483365, 35483499, 35483510],
                          [       0,       25,       25,       36]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19      35483340 ????????????????????????????????????????????????????????????
                  0 |||||||||||||||||||||||||-----------------------------------
hg19_dna          0 ?????????????????????????-----------------------------------

chr19      35483400 ????????????????????????????????????????????????????????????
                 60 ------------------------------------------------------------
hg19_dna         25 ------------------------------------------------------------

chr19      35483460 ?????????????????????????????????????????????????? 35483510
                120 ---------------------------------------|||||||||||      170
hg19_dna         25 ---------------------------------------???????????       36
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	35483340	35483510	hg19_dna	890	+	35483340	35483510	0	2	25,11,	0,159,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 134 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 134:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 134:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 134:
                open_internal_deletions = 1,
                extend_internal_deletions = 133;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 134)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 134)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 134)
        self.assertEqual(counts.gaps, 134)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr18")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[23891310, 23891349],
                          [       0,       39]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr18      23891310 ??????????????????????????????????????? 23891349
                  0 |||||||||||||||||||||||||||||||||||||||       39
hg19_dna          0 ???????????????????????????????????????       39
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr18	23891310	23891349	hg19_dna	1000	+	23891310	23891349	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 930)
        self.assertEqual(alignment.shape, (2, 28))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr18")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[43252217, 43252245],
                          [       0,       28]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr18      43252217 ???????????????????????????? 43252245
                  0 ||||||||||||||||||||||||||||       28
hg19_dna          0 ????????????????????????????       28
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr18	43252217	43252245	hg19_dna	930	+	43252217	43252245	0	1	28,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (28 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 28:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 28)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 912)
        self.assertEqual(alignment.shape, (2, 51))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[52759147, 52759154, 52759160, 52759198],
                          [       0,        7,        7,       45]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      52759147 ??????????????????????????????????????????????????? 52759198
                  0 |||||||------||||||||||||||||||||||||||||||||||||||       51
hg19_dna          0 ???????------??????????????????????????????????????       45
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	52759147	52759198	hg19_dna	912	+	52759147	52759198	0	2	7,38,	0,13,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (45 aligned letters; 0 identities; 0 mismatches; 6 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 45:
        identities = 0,
        mismatches = 0.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 1,
                extend_internal_deletions = 5;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 6)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 6)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 6)
        self.assertEqual(counts.gaps, 6)
        self.assertEqual(counts.aligned, 45)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 50))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[1207056, 1207106],
                          [      0,      50]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1        1207056 ?????????????????????????????????????????????????? 1207106
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||      50
hg19_dna          0 ??????????????????????????????????????????????????      50
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	1207056	1207106	hg19_dna	1000	+	1207056	1207106	0	1	50,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (50 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 50:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 50)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 824)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[61700837, 61700871],
                          [       0,       34]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       61700837 ?????????????????????????????????? 61700871
                  0 ||||||||||||||||||||||||||||||||||       34
hg19_dna          0 ??????????????????????????????????       34
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	61700837	61700871	hg19_dna	824	+	61700837	61700871	0	1	34,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (34 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 34:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 34)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 572)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr4")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[37558157, 37558167, 37558173, 37558191],
                          [      28,       18,       18,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr4       37558157 ?????????????????????????????????? 37558191
                  0 ||||||||||------||||||||||||||||||       34
hg19_dna         28 ??????????------??????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr4	37558157	37558191	hg19_dna	572	-	37558157	37558191	0	2	10,18,	0,16,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (28 aligned letters; 0 identities; 0 mismatches; 6 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 28:
        identities = 0,
        mismatches = 0.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 1,
                extend_internal_deletions = 5;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 6)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 6)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 6)
        self.assertEqual(counts.gaps, 6)
        self.assertEqual(counts.aligned, 28)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 892)
        self.assertEqual(alignment.shape, (2, 37))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr22")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[48997405, 48997442],
                          [      37,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr22      48997405 ????????????????????????????????????? 48997442
                  0 |||||||||||||||||||||||||||||||||||||       37
hg19_dna         37 ?????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr22	48997405	48997442	hg19_dna	892	-	48997405	48997442	0	1	37,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (37 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 37:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 37)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 946)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[120641740, 120641776],
                          [       36,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2      120641740 ???????????????????????????????????? 120641776
                  0 ||||||||||||||||||||||||||||||||||||        36
hg19_dna         36 ????????????????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	120641740	120641776	hg19_dna	946	-	120641740	120641776	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[54017130, 54017169],
                          [      39,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19      54017130 ??????????????????????????????????????? 54017169
                  0 |||||||||||||||||||||||||||||||||||||||       39
hg19_dna         39 ???????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	54017130	54017169	hg19_dna	1000	-	54017130	54017169	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 848)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[553742, 553781],
                          [    39,      0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19        553742 ??????????????????????????????????????? 553781
                  0 |||||||||||||||||||||||||||||||||||||||     39
hg19_dna         39 ???????????????????????????????????????      0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	553742	553781	hg19_dna	848	-	553742	553781	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 834)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr10")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[99388555, 99388591],
                          [      36,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr10      99388555 ???????????????????????????????????? 99388591
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna         36 ????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr10	99388555	99388591	hg19_dna	834	-	99388555	99388591	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 920)
        self.assertEqual(alignment.shape, (2, 25))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr10")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[112178171, 112178196],
                          [       25,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr10     112178171 ????????????????????????? 112178196
                  0 |||||||||||||||||||||||||        25
hg19_dna         25 ?????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr10	112178171	112178196	hg19_dna	920	-	112178171	112178196	0	1	25,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (25 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 25:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 25)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 946)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[39368490, 39368526],
                          [      36,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       39368490 ???????????????????????????????????? 39368526
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna         36 ????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	39368490	39368526	hg19_dna	946	-	39368490	39368526	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 942)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[220325687, 220325721],
                          [       34,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1      220325687 ?????????????????????????????????? 220325721
                  0 ||||||||||||||||||||||||||||||||||        34
hg19_dna         34 ??????????????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	220325687	220325721	hg19_dna	942	-	220325687	220325721	0	1	34,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (34 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 34:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 34)
        self.assertRaises(StopIteration, next, alignments)

    def test_writing_psl_34_001(self):
        """Test writing the alignments in psl_34_001.bed."""
        path = "Blat/psl_34_001.bed"
        with open(path) as stream:
            original_data = stream.read()
        alignments = Align.parse(path, "bed")
        stream = StringIO()
        n = Align.write(alignments, stream, "bed")
        self.assertEqual(n, 22)
        stream.seek(0)
        written_data = stream.read()
        stream.close()
        self.assertEqual(original_data, written_data)

    def test_reading_psl_34_003(self):
        """Test parsing psl_34_003.bed."""
        path = "Blat/psl_34_003.bed"
        alignments = Align.parse(path, "bed")
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 16))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr4")
        self.assertEqual(alignment.query.id, "hg18_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[61646095, 61646111],
                          [       0,       16]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr4       61646095 ???????????????? 61646111
                  0 ||||||||||||||||       16
hg18_dna          0 ????????????????       16
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr4	61646095	61646111	hg18_dna	1000	+	61646095	61646111	0	1	16,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (16 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 16:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 16)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 33))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg18_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[10271783, 10271816],
                          [       0,       33]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       10271783 ????????????????????????????????? 10271816
                  0 |||||||||||||||||||||||||||||||||       33
hg18_dna          0 ?????????????????????????????????       33
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	10271783	10271816	hg18_dna	1000	+	10271783	10271816	0	1	33,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (33 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 33:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 33)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 17))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg18_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[53575980, 53575997],
                          [      17,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2       53575980 ????????????????? 53575997
                  0 |||||||||||||||||       17
hg18_dna         17 ?????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	53575980	53575997	hg18_dna	1000	-	53575980	53575997	0	1	17,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (17 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 17:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 17)
        self.assertRaises(StopIteration, next, alignments)

    def test_writing_psl_34_003(self):
        """Test writing the alignments in psl_34_003.bed."""
        path = "Blat/psl_34_003.bed"
        with open(path) as stream:
            original_data = stream.read()
        alignments = Align.parse(path, "bed")
        stream = StringIO()
        n = Align.write(alignments, stream, "bed")
        self.assertEqual(n, 3)
        stream.seek(0)
        written_data = stream.read()
        stream.close()
        self.assertEqual(original_data, written_data)

    def test_reading_psl_34_004(self):
        """Test parsing psl_34_004.bed."""
        path = "Blat/psl_34_004.bed"
        alignments = Align.parse(path, "bed")
        alignment = next(alignments)
        self.assertEqual(alignment.score, 854)
        self.assertEqual(alignment.shape, (2, 41))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr9")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[85737865, 85737906],
                          [       0,       41]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr9       85737865 ????????????????????????????????????????? 85737906
                  0 |||||||||||||||||||||||||||||||||||||||||       41
hg19_dna          0 ?????????????????????????????????????????       41
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr9	85737865	85737906	hg19_dna	854	+	85737865	85737906	0	1	41,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (41 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 41:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 41)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 41))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr8")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[95160479, 95160520],
                          [       0,       41]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr8       95160479 ????????????????????????????????????????? 95160520
                  0 |||||||||||||||||||||||||||||||||||||||||       41
hg19_dna          0 ?????????????????????????????????????????       41
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr8	95160479	95160520	hg19_dna	1000	+	95160479	95160520	0	1	41,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (41 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 41:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 41)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 834)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr22")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[42144400, 42144436],
                          [       0,       36]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr22      42144400 ???????????????????????????????????? 42144436
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna          0 ????????????????????????????????????       36
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr22	42144400	42144436	hg19_dna	834	+	42144400	42144436	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 682)
        self.assertEqual(alignment.shape, (2, 44))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[183925984, 183925990, 183926028],
                          [        0,         6,        44]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2      183925984 ???????????????????????????????????????????? 183926028
                  0 ||||||||||||||||||||||||||||||||||||||||||||        44
hg19_dna          0 ????????????????????????????????????????????        44
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	183925984	183926028	hg19_dna	682	+	183925984	183926028	0	2	6,38,	0,6,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (44 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 44:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 44)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 890)
        self.assertEqual(alignment.shape, (2, 170))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[35483340, 35483365, 35483499, 35483510],
                          [       0,       25,       25,       36]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19      35483340 ????????????????????????????????????????????????????????????
                  0 |||||||||||||||||||||||||-----------------------------------
hg19_dna          0 ?????????????????????????-----------------------------------

chr19      35483400 ????????????????????????????????????????????????????????????
                 60 ------------------------------------------------------------
hg19_dna         25 ------------------------------------------------------------

chr19      35483460 ?????????????????????????????????????????????????? 35483510
                120 ---------------------------------------|||||||||||      170
hg19_dna         25 ---------------------------------------???????????       36
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	35483340	35483510	hg19_dna	890	+	35483340	35483510	0	2	25,11,	0,159,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 134 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 134:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 134:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 134:
                open_internal_deletions = 1,
                extend_internal_deletions = 133;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 134)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 134)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 134)
        self.assertEqual(counts.gaps, 134)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr18")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[23891310, 23891349],
                          [       0,       39]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr18      23891310 ??????????????????????????????????????? 23891349
                  0 |||||||||||||||||||||||||||||||||||||||       39
hg19_dna          0 ???????????????????????????????????????       39
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr18	23891310	23891349	hg19_dna	1000	+	23891310	23891349	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 930)
        self.assertEqual(alignment.shape, (2, 28))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr18")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[43252217, 43252245],
                          [       0,       28]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr18      43252217 ???????????????????????????? 43252245
                  0 ||||||||||||||||||||||||||||       28
hg19_dna          0 ????????????????????????????       28
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr18	43252217	43252245	hg19_dna	930	+	43252217	43252245	0	1	28,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (28 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 28:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 28)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 912)
        self.assertEqual(alignment.shape, (2, 51))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[52759147, 52759154, 52759160, 52759198],
                          [       0,        7,        7,       45]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      52759147 ??????????????????????????????????????????????????? 52759198
                  0 |||||||------||||||||||||||||||||||||||||||||||||||       51
hg19_dna          0 ???????------??????????????????????????????????????       45
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	52759147	52759198	hg19_dna	912	+	52759147	52759198	0	2	7,38,	0,13,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (45 aligned letters; 0 identities; 0 mismatches; 6 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 45:
        identities = 0,
        mismatches = 0.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 1,
                extend_internal_deletions = 5;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 6)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 6)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 6)
        self.assertEqual(counts.gaps, 6)
        self.assertEqual(counts.aligned, 45)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 50))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[1207056, 1207106],
                          [      0,      50]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1        1207056 ?????????????????????????????????????????????????? 1207106
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||      50
hg19_dna          0 ??????????????????????????????????????????????????      50
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	1207056	1207106	hg19_dna	1000	+	1207056	1207106	0	1	50,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (50 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 50:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 50)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 824)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[61700837, 61700871],
                          [       0,       34]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       61700837 ?????????????????????????????????? 61700871
                  0 ||||||||||||||||||||||||||||||||||       34
hg19_dna          0 ??????????????????????????????????       34
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	61700837	61700871	hg19_dna	824	+	61700837	61700871	0	1	34,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (34 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 34:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 34)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 572)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr4")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[37558157, 37558167, 37558173, 37558191],
                          [      28,       18,       18,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr4       37558157 ?????????????????????????????????? 37558191
                  0 ||||||||||------||||||||||||||||||       34
hg19_dna         28 ??????????------??????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr4	37558157	37558191	hg19_dna	572	-	37558157	37558191	0	2	10,18,	0,16,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (28 aligned letters; 0 identities; 0 mismatches; 6 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 28:
        identities = 0,
        mismatches = 0.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 1,
                extend_internal_deletions = 5;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 6)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 6)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 6)
        self.assertEqual(counts.gaps, 6)
        self.assertEqual(counts.aligned, 28)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 892)
        self.assertEqual(alignment.shape, (2, 37))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr22")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[48997405, 48997442],
                          [      37,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr22      48997405 ????????????????????????????????????? 48997442
                  0 |||||||||||||||||||||||||||||||||||||       37
hg19_dna         37 ?????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr22	48997405	48997442	hg19_dna	892	-	48997405	48997442	0	1	37,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (37 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 37:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 37)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 946)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[120641740, 120641776],
                          [       36,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2      120641740 ???????????????????????????????????? 120641776
                  0 ||||||||||||||||||||||||||||||||||||        36
hg19_dna         36 ????????????????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	120641740	120641776	hg19_dna	946	-	120641740	120641776	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[54017130, 54017169],
                          [      39,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19      54017130 ??????????????????????????????????????? 54017169
                  0 |||||||||||||||||||||||||||||||||||||||       39
hg19_dna         39 ???????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	54017130	54017169	hg19_dna	1000	-	54017130	54017169	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 848)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[553742, 553781],
                          [    39,      0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19        553742 ??????????????????????????????????????? 553781
                  0 |||||||||||||||||||||||||||||||||||||||     39
hg19_dna         39 ???????????????????????????????????????      0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	553742	553781	hg19_dna	848	-	553742	553781	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 834)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr10")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[99388555, 99388591],
                          [      36,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr10      99388555 ???????????????????????????????????? 99388591
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna         36 ????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr10	99388555	99388591	hg19_dna	834	-	99388555	99388591	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 920)
        self.assertEqual(alignment.shape, (2, 25))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr10")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[112178171, 112178196],
                          [       25,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr10     112178171 ????????????????????????? 112178196
                  0 |||||||||||||||||||||||||        25
hg19_dna         25 ?????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr10	112178171	112178196	hg19_dna	920	-	112178171	112178196	0	1	25,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (25 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 25:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 25)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 946)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[39368490, 39368526],
                          [      36,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       39368490 ???????????????????????????????????? 39368526
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna         36 ????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	39368490	39368526	hg19_dna	946	-	39368490	39368526	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 942)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[220325687, 220325721],
                          [       34,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1      220325687 ?????????????????????????????????? 220325721
                  0 ||||||||||||||||||||||||||||||||||        34
hg19_dna         34 ??????????????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	220325687	220325721	hg19_dna	942	-	220325687	220325721	0	1	34,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (34 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 34:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 34)
        self.assertRaises(StopIteration, next, alignments)

    def test_writing_psl_34_004(self):
        """Test writing the alignments in psl_34_004.bed."""
        path = "Blat/psl_34_004.bed"
        with open(path) as stream:
            original_data = stream.read()
        alignments = Align.parse(path, "bed")
        stream = StringIO()
        n = Align.write(alignments, stream, "bed")
        self.assertEqual(n, 19)
        stream.seek(0)
        written_data = stream.read()
        stream.close()
        self.assertEqual(original_data, written_data)

    def test_reading_psl_34_005(self):
        """Test parsing psl_34_005.bed."""
        path = "Blat/psl_34_005.bed"
        alignments = Align.parse(path, "bed")
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 16))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr4")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[61646095, 61646111],
                          [       0,       16]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr4       61646095 ???????????????? 61646111
                  0 ||||||||||||||||       16
hg19_dna          0 ????????????????       16
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr4	61646095	61646111	hg19_dna	1000	+	61646095	61646111	0	1	16,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (16 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 16:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 16)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 33))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[10271783, 10271816],
                          [       0,       33]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       10271783 ????????????????????????????????? 10271816
                  0 |||||||||||||||||||||||||||||||||       33
hg19_dna          0 ?????????????????????????????????       33
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	10271783	10271816	hg19_dna	1000	+	10271783	10271816	0	1	33,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (33 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 33:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 33)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 17))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[53575980, 53575997],
                          [      17,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2       53575980 ????????????????? 53575997
                  0 |||||||||||||||||       17
hg19_dna         17 ?????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	53575980	53575997	hg19_dna	1000	-	53575980	53575997	0	1	17,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (17 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 17:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 17)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 854)
        self.assertEqual(alignment.shape, (2, 41))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr9")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[85737865, 85737906],
                          [       0,       41]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr9       85737865 ????????????????????????????????????????? 85737906
                  0 |||||||||||||||||||||||||||||||||||||||||       41
hg19_dna          0 ?????????????????????????????????????????       41
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr9	85737865	85737906	hg19_dna	854	+	85737865	85737906	0	1	41,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (41 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 41:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 41)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 41))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr8")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[95160479, 95160520],
                          [       0,       41]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr8       95160479 ????????????????????????????????????????? 95160520
                  0 |||||||||||||||||||||||||||||||||||||||||       41
hg19_dna          0 ?????????????????????????????????????????       41
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr8	95160479	95160520	hg19_dna	1000	+	95160479	95160520	0	1	41,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (41 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 41:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 41)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 834)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr22")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[42144400, 42144436],
                          [       0,       36]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr22      42144400 ???????????????????????????????????? 42144436
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna          0 ????????????????????????????????????       36
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr22	42144400	42144436	hg19_dna	834	+	42144400	42144436	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 682)
        self.assertEqual(alignment.shape, (2, 44))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[183925984, 183925990, 183926028],
                          [        0,         6,        44]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2      183925984 ???????????????????????????????????????????? 183926028
                  0 ||||||||||||||||||||||||||||||||||||||||||||        44
hg19_dna          0 ????????????????????????????????????????????        44
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	183925984	183926028	hg19_dna	682	+	183925984	183926028	0	2	6,38,	0,6,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (44 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 44:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 44)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 890)
        self.assertEqual(alignment.shape, (2, 170))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[35483340, 35483365, 35483499, 35483510],
                          [       0,       25,       25,       36]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19      35483340 ????????????????????????????????????????????????????????????
                  0 |||||||||||||||||||||||||-----------------------------------
hg19_dna          0 ?????????????????????????-----------------------------------

chr19      35483400 ????????????????????????????????????????????????????????????
                 60 ------------------------------------------------------------
hg19_dna         25 ------------------------------------------------------------

chr19      35483460 ?????????????????????????????????????????????????? 35483510
                120 ---------------------------------------|||||||||||      170
hg19_dna         25 ---------------------------------------???????????       36
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	35483340	35483510	hg19_dna	890	+	35483340	35483510	0	2	25,11,	0,159,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 134 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 134:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 134:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 134:
                open_internal_deletions = 1,
                extend_internal_deletions = 133;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 134)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 134)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 134)
        self.assertEqual(counts.gaps, 134)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr18")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[23891310, 23891349],
                          [       0,       39]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr18      23891310 ??????????????????????????????????????? 23891349
                  0 |||||||||||||||||||||||||||||||||||||||       39
hg19_dna          0 ???????????????????????????????????????       39
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr18	23891310	23891349	hg19_dna	1000	+	23891310	23891349	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 930)
        self.assertEqual(alignment.shape, (2, 28))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr18")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[43252217, 43252245],
                          [       0,       28]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr18      43252217 ???????????????????????????? 43252245
                  0 ||||||||||||||||||||||||||||       28
hg19_dna          0 ????????????????????????????       28
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr18	43252217	43252245	hg19_dna	930	+	43252217	43252245	0	1	28,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (28 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 28:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 28)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 912)
        self.assertEqual(alignment.shape, (2, 51))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[52759147, 52759154, 52759160, 52759198],
                          [       0,        7,        7,       45]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      52759147 ??????????????????????????????????????????????????? 52759198
                  0 |||||||------||||||||||||||||||||||||||||||||||||||       51
hg19_dna          0 ???????------??????????????????????????????????????       45
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	52759147	52759198	hg19_dna	912	+	52759147	52759198	0	2	7,38,	0,13,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (45 aligned letters; 0 identities; 0 mismatches; 6 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 45:
        identities = 0,
        mismatches = 0.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 1,
                extend_internal_deletions = 5;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 6)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 6)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 6)
        self.assertEqual(counts.gaps, 6)
        self.assertEqual(counts.aligned, 45)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 50))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[1207056, 1207106],
                          [      0,      50]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1        1207056 ?????????????????????????????????????????????????? 1207106
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||      50
hg19_dna          0 ??????????????????????????????????????????????????      50
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	1207056	1207106	hg19_dna	1000	+	1207056	1207106	0	1	50,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (50 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 50:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 50)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 824)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[61700837, 61700871],
                          [       0,       34]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       61700837 ?????????????????????????????????? 61700871
                  0 ||||||||||||||||||||||||||||||||||       34
hg19_dna          0 ??????????????????????????????????       34
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	61700837	61700871	hg19_dna	824	+	61700837	61700871	0	1	34,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (34 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 34:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 34)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 572)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr4")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[37558157, 37558167, 37558173, 37558191],
                          [      28,       18,       18,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr4       37558157 ?????????????????????????????????? 37558191
                  0 ||||||||||------||||||||||||||||||       34
hg19_dna         28 ??????????------??????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr4	37558157	37558191	hg19_dna	572	-	37558157	37558191	0	2	10,18,	0,16,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (28 aligned letters; 0 identities; 0 mismatches; 6 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 28:
        identities = 0,
        mismatches = 0.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 1,
                extend_internal_deletions = 5;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 6)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 6)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 6)
        self.assertEqual(counts.gaps, 6)
        self.assertEqual(counts.aligned, 28)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 892)
        self.assertEqual(alignment.shape, (2, 37))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr22")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[48997405, 48997442],
                          [      37,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr22      48997405 ????????????????????????????????????? 48997442
                  0 |||||||||||||||||||||||||||||||||||||       37
hg19_dna         37 ?????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr22	48997405	48997442	hg19_dna	892	-	48997405	48997442	0	1	37,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (37 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 37:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 37)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 946)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[120641740, 120641776],
                          [       36,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr2      120641740 ???????????????????????????????????? 120641776
                  0 ||||||||||||||||||||||||||||||||||||        36
hg19_dna         36 ????????????????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr2	120641740	120641776	hg19_dna	946	-	120641740	120641776	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[54017130, 54017169],
                          [      39,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19      54017130 ??????????????????????????????????????? 54017169
                  0 |||||||||||||||||||||||||||||||||||||||       39
hg19_dna         39 ???????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	54017130	54017169	hg19_dna	1000	-	54017130	54017169	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 848)
        self.assertEqual(alignment.shape, (2, 39))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr19")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[553742, 553781],
                          [    39,      0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr19        553742 ??????????????????????????????????????? 553781
                  0 |||||||||||||||||||||||||||||||||||||||     39
hg19_dna         39 ???????????????????????????????????????      0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr19	553742	553781	hg19_dna	848	-	553742	553781	0	1	39,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (39 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 39:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 39)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 834)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr10")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[99388555, 99388591],
                          [      36,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr10      99388555 ???????????????????????????????????? 99388591
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna         36 ????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr10	99388555	99388591	hg19_dna	834	-	99388555	99388591	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 920)
        self.assertEqual(alignment.shape, (2, 25))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr10")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[112178171, 112178196],
                          [       25,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr10     112178171 ????????????????????????? 112178196
                  0 |||||||||||||||||||||||||        25
hg19_dna         25 ?????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr10	112178171	112178196	hg19_dna	920	-	112178171	112178196	0	1	25,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (25 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 25:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 25)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 946)
        self.assertEqual(alignment.shape, (2, 36))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[39368490, 39368526],
                          [      36,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1       39368490 ???????????????????????????????????? 39368526
                  0 ||||||||||||||||||||||||||||||||||||       36
hg19_dna         36 ????????????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	39368490	39368526	hg19_dna	946	-	39368490	39368526	0	1	36,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (36 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 36:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 36)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 942)
        self.assertEqual(alignment.shape, (2, 34))
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "hg19_dna")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[220325687, 220325721],
                          [       34,         0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr1      220325687 ?????????????????????????????????? 220325721
                  0 ||||||||||||||||||||||||||||||||||        34
hg19_dna         34 ??????????????????????????????????         0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr1	220325687	220325721	hg19_dna	942	-	220325687	220325721	0	1	34,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (34 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 34:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 34)
        self.assertRaises(StopIteration, next, alignments)

    def test_writing_psl_34_005(self):
        """Test writing the alignments in psl_34_005.bed."""
        path = "Blat/psl_34_005.bed"
        with open(path) as stream:
            original_data = stream.read()
        alignments = Align.parse(path, "bed")
        stream = StringIO()
        n = Align.write(alignments, stream, "bed")
        self.assertEqual(n, 22)
        stream.seek(0)
        written_data = stream.read()
        stream.close()
        self.assertEqual(original_data, written_data)


class TestAlign_dnax_prot(unittest.TestCase):
    def test_reading_psl_35_001(self):
        """Test parsing psl_35_001.bed."""
        path = "Blat/psl_35_001.bed"
        alignments = Align.parse(path, "bed")
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[75566694, 75566850],
                          [       0,      156]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      75566694 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.         0 ????????????????????????????????????????????????????????????

chr13      75566754 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.        60 ????????????????????????????????????????????????????????????

chr13      75566814 ???????????????????????????????????? 75566850
                120 ||||||||||||||||||||||||||||||||||||      156
CAG33136.       120 ????????????????????????????????????      156
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	75566694	75566850	CAG33136.1	1000	+	75566694	75566850	0	1	156,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (156 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 156:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 156)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[75560749, 75560881],
                          [       0,      132]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      75560749 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.         0 ????????????????????????????????????????????????????????????

chr13      75560809 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.        60 ????????????????????????????????????????????????????????????

chr13      75560869 ???????????? 75560881
                120 ||||||||||||      132
CAG33136.       120 ????????????      132
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	75560749	75560881	CAG33136.1	1000	+	75560749	75560881	0	1	132,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (132 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 132:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 132)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 986)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[75549820, 75549865, 75567225, 75567312],
                          [       0,       45,       45,      132]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      75549820 ????????????????????????????????????????????????????????????
                  0 |||||||||||||||||||||||||||||||||||||||||||||---------------
CAG33136.         0 ?????????????????????????????????????????????---------------

chr13      75549880 ????????????????????????????????????????????????????????????
                 60 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75549940 ????????????????????????????????????????????????????????????
                120 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550000 ????????????????????????????????????????????????????????????
                180 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550060 ????????????????????????????????????????????????????????????
                240 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550120 ????????????????????????????????????????????????????????????
                300 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550180 ????????????????????????????????????????????????????????????
                360 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550240 ????????????????????????????????????????????????????????????
                420 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550300 ????????????????????????????????????????????????????????????
                480 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550360 ????????????????????????????????????????????????????????????
                540 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550420 ????????????????????????????????????????????????????????????
                600 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550480 ????????????????????????????????????????????????????????????
                660 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550540 ????????????????????????????????????????????????????????????
                720 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550600 ????????????????????????????????????????????????????????????
                780 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550660 ????????????????????????????????????????????????????????????
                840 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550720 ????????????????????????????????????????????????????????????
                900 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550780 ????????????????????????????????????????????????????????????
                960 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550840 ????????????????????????????????????????????????????????????
               1020 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550900 ????????????????????????????????????????????????????????????
               1080 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75550960 ????????????????????????????????????????????????????????????
               1140 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551020 ????????????????????????????????????????????????????????????
               1200 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551080 ????????????????????????????????????????????????????????????
               1260 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551140 ????????????????????????????????????????????????????????????
               1320 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551200 ????????????????????????????????????????????????????????????
               1380 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551260 ????????????????????????????????????????????????????????????
               1440 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551320 ????????????????????????????????????????????????????????????
               1500 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551380 ????????????????????????????????????????????????????????????
               1560 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551440 ????????????????????????????????????????????????????????????
               1620 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551500 ????????????????????????????????????????????????????????????
               1680 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551560 ????????????????????????????????????????????????????????????
               1740 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551620 ????????????????????????????????????????????????????????????
               1800 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551680 ????????????????????????????????????????????????????????????
               1860 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551740 ????????????????????????????????????????????????????????????
               1920 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551800 ????????????????????????????????????????????????????????????
               1980 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551860 ????????????????????????????????????????????????????????????
               2040 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551920 ????????????????????????????????????????????????????????????
               2100 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75551980 ????????????????????????????????????????????????????????????
               2160 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552040 ????????????????????????????????????????????????????????????
               2220 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552100 ????????????????????????????????????????????????????????????
               2280 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552160 ????????????????????????????????????????????????????????????
               2340 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552220 ????????????????????????????????????????????????????????????
               2400 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552280 ????????????????????????????????????????????????????????????
               2460 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552340 ????????????????????????????????????????????????????????????
               2520 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552400 ????????????????????????????????????????????????????????????
               2580 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552460 ????????????????????????????????????????????????????????????
               2640 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552520 ????????????????????????????????????????????????????????????
               2700 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552580 ????????????????????????????????????????????????????????????
               2760 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552640 ????????????????????????????????????????????????????????????
               2820 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552700 ????????????????????????????????????????????????????????????
               2880 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552760 ????????????????????????????????????????????????????????????
               2940 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552820 ????????????????????????????????????????????????????????????
               3000 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552880 ????????????????????????????????????????????????????????????
               3060 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75552940 ????????????????????????????????????????????????????????????
               3120 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553000 ????????????????????????????????????????????????????????????
               3180 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553060 ????????????????????????????????????????????????????????????
               3240 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553120 ????????????????????????????????????????????????????????????
               3300 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553180 ????????????????????????????????????????????????????????????
               3360 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553240 ????????????????????????????????????????????????????????????
               3420 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553300 ????????????????????????????????????????????????????????????
               3480 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553360 ????????????????????????????????????????????????????????????
               3540 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553420 ????????????????????????????????????????????????????????????
               3600 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553480 ????????????????????????????????????????????????????????????
               3660 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553540 ????????????????????????????????????????????????????????????
               3720 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553600 ????????????????????????????????????????????????????????????
               3780 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553660 ????????????????????????????????????????????????????????????
               3840 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553720 ????????????????????????????????????????????????????????????
               3900 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553780 ????????????????????????????????????????????????????????????
               3960 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553840 ????????????????????????????????????????????????????????????
               4020 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553900 ????????????????????????????????????????????????????????????
               4080 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75553960 ????????????????????????????????????????????????????????????
               4140 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554020 ????????????????????????????????????????????????????????????
               4200 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554080 ????????????????????????????????????????????????????????????
               4260 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554140 ????????????????????????????????????????????????????????????
               4320 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554200 ????????????????????????????????????????????????????????????
               4380 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554260 ????????????????????????????????????????????????????????????
               4440 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554320 ????????????????????????????????????????????????????????????
               4500 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554380 ????????????????????????????????????????????????????????????
               4560 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554440 ????????????????????????????????????????????????????????????
               4620 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554500 ????????????????????????????????????????????????????????????
               4680 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554560 ????????????????????????????????????????????????????????????
               4740 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554620 ????????????????????????????????????????????????????????????
               4800 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554680 ????????????????????????????????????????????????????????????
               4860 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554740 ????????????????????????????????????????????????????????????
               4920 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554800 ????????????????????????????????????????????????????????????
               4980 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554860 ????????????????????????????????????????????????????????????
               5040 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554920 ????????????????????????????????????????????????????????????
               5100 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75554980 ????????????????????????????????????????????????????????????
               5160 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555040 ????????????????????????????????????????????????????????????
               5220 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555100 ????????????????????????????????????????????????????????????
               5280 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555160 ????????????????????????????????????????????????????????????
               5340 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555220 ????????????????????????????????????????????????????????????
               5400 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555280 ????????????????????????????????????????????????????????????
               5460 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555340 ????????????????????????????????????????????????????????????
               5520 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555400 ????????????????????????????????????????????????????????????
               5580 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555460 ????????????????????????????????????????????????????????????
               5640 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555520 ????????????????????????????????????????????????????????????
               5700 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555580 ????????????????????????????????????????????????????????????
               5760 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555640 ????????????????????????????????????????????????????????????
               5820 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555700 ????????????????????????????????????????????????????????????
               5880 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555760 ????????????????????????????????????????????????????????????
               5940 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555820 ????????????????????????????????????????????????????????????
               6000 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555880 ????????????????????????????????????????????????????????????
               6060 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75555940 ????????????????????????????????????????????????????????????
               6120 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556000 ????????????????????????????????????????????????????????????
               6180 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556060 ????????????????????????????????????????????????????????????
               6240 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556120 ????????????????????????????????????????????????????????????
               6300 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556180 ????????????????????????????????????????????????????????????
               6360 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556240 ????????????????????????????????????????????????????????????
               6420 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556300 ????????????????????????????????????????????????????????????
               6480 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556360 ????????????????????????????????????????????????????????????
               6540 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556420 ????????????????????????????????????????????????????????????
               6600 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556480 ????????????????????????????????????????????????????????????
               6660 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556540 ????????????????????????????????????????????????????????????
               6720 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556600 ????????????????????????????????????????????????????????????
               6780 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556660 ????????????????????????????????????????????????????????????
               6840 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556720 ????????????????????????????????????????????????????????????
               6900 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556780 ????????????????????????????????????????????????????????????
               6960 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556840 ????????????????????????????????????????????????????????????
               7020 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556900 ????????????????????????????????????????????????????????????
               7080 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75556960 ????????????????????????????????????????????????????????????
               7140 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557020 ????????????????????????????????????????????????????????????
               7200 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557080 ????????????????????????????????????????????????????????????
               7260 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557140 ????????????????????????????????????????????????????????????
               7320 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557200 ????????????????????????????????????????????????????????????
               7380 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557260 ????????????????????????????????????????????????????????????
               7440 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557320 ????????????????????????????????????????????????????????????
               7500 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557380 ????????????????????????????????????????????????????????????
               7560 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557440 ????????????????????????????????????????????????????????????
               7620 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557500 ????????????????????????????????????????????????????????????
               7680 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557560 ????????????????????????????????????????????????????????????
               7740 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557620 ????????????????????????????????????????????????????????????
               7800 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557680 ????????????????????????????????????????????????????????????
               7860 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557740 ????????????????????????????????????????????????????????????
               7920 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557800 ????????????????????????????????????????????????????????????
               7980 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557860 ????????????????????????????????????????????????????????????
               8040 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557920 ????????????????????????????????????????????????????????????
               8100 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75557980 ????????????????????????????????????????????????????????????
               8160 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558040 ????????????????????????????????????????????????????????????
               8220 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558100 ????????????????????????????????????????????????????????????
               8280 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558160 ????????????????????????????????????????????????????????????
               8340 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558220 ????????????????????????????????????????????????????????????
               8400 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558280 ????????????????????????????????????????????????????????????
               8460 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558340 ????????????????????????????????????????????????????????????
               8520 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558400 ????????????????????????????????????????????????????????????
               8580 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558460 ????????????????????????????????????????????????????????????
               8640 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558520 ????????????????????????????????????????????????????????????
               8700 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558580 ????????????????????????????????????????????????????????????
               8760 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558640 ????????????????????????????????????????????????????????????
               8820 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558700 ????????????????????????????????????????????????????????????
               8880 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558760 ????????????????????????????????????????????????????????????
               8940 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558820 ????????????????????????????????????????????????????????????
               9000 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558880 ????????????????????????????????????????????????????????????
               9060 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75558940 ????????????????????????????????????????????????????????????
               9120 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559000 ????????????????????????????????????????????????????????????
               9180 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559060 ????????????????????????????????????????????????????????????
               9240 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559120 ????????????????????????????????????????????????????????????
               9300 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559180 ????????????????????????????????????????????????????????????
               9360 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559240 ????????????????????????????????????????????????????????????
               9420 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559300 ????????????????????????????????????????????????????????????
               9480 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559360 ????????????????????????????????????????????????????????????
               9540 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559420 ????????????????????????????????????????????????????????????
               9600 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559480 ????????????????????????????????????????????????????????????
               9660 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559540 ????????????????????????????????????????????????????????????
               9720 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559600 ????????????????????????????????????????????????????????????
               9780 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559660 ????????????????????????????????????????????????????????????
               9840 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559720 ????????????????????????????????????????????????????????????
               9900 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559780 ????????????????????????????????????????????????????????????
               9960 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559840 ????????????????????????????????????????????????????????????
              10020 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559900 ????????????????????????????????????????????????????????????
              10080 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75559960 ????????????????????????????????????????????????????????????
              10140 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560020 ????????????????????????????????????????????????????????????
              10200 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560080 ????????????????????????????????????????????????????????????
              10260 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560140 ????????????????????????????????????????????????????????????
              10320 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560200 ????????????????????????????????????????????????????????????
              10380 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560260 ????????????????????????????????????????????????????????????
              10440 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560320 ????????????????????????????????????????????????????????????
              10500 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560380 ????????????????????????????????????????????????????????????
              10560 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560440 ????????????????????????????????????????????????????????????
              10620 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560500 ????????????????????????????????????????????????????????????
              10680 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560560 ????????????????????????????????????????????????????????????
              10740 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560620 ????????????????????????????????????????????????????????????
              10800 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560680 ????????????????????????????????????????????????????????????
              10860 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560740 ????????????????????????????????????????????????????????????
              10920 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560800 ????????????????????????????????????????????????????????????
              10980 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560860 ????????????????????????????????????????????????????????????
              11040 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560920 ????????????????????????????????????????????????????????????
              11100 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75560980 ????????????????????????????????????????????????????????????
              11160 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561040 ????????????????????????????????????????????????????????????
              11220 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561100 ????????????????????????????????????????????????????????????
              11280 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561160 ????????????????????????????????????????????????????????????
              11340 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561220 ????????????????????????????????????????????????????????????
              11400 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561280 ????????????????????????????????????????????????????????????
              11460 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561340 ????????????????????????????????????????????????????????????
              11520 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561400 ????????????????????????????????????????????????????????????
              11580 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561460 ????????????????????????????????????????????????????????????
              11640 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561520 ????????????????????????????????????????????????????????????
              11700 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561580 ????????????????????????????????????????????????????????????
              11760 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561640 ????????????????????????????????????????????????????????????
              11820 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561700 ????????????????????????????????????????????????????????????
              11880 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561760 ????????????????????????????????????????????????????????????
              11940 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561820 ????????????????????????????????????????????????????????????
              12000 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561880 ????????????????????????????????????????????????????????????
              12060 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75561940 ????????????????????????????????????????????????????????????
              12120 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562000 ????????????????????????????????????????????????????????????
              12180 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562060 ????????????????????????????????????????????????????????????
              12240 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562120 ????????????????????????????????????????????????????????????
              12300 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562180 ????????????????????????????????????????????????????????????
              12360 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562240 ????????????????????????????????????????????????????????????
              12420 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562300 ????????????????????????????????????????????????????????????
              12480 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562360 ????????????????????????????????????????????????????????????
              12540 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562420 ????????????????????????????????????????????????????????????
              12600 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562480 ????????????????????????????????????????????????????????????
              12660 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562540 ????????????????????????????????????????????????????????????
              12720 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562600 ????????????????????????????????????????????????????????????
              12780 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562660 ????????????????????????????????????????????????????????????
              12840 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562720 ????????????????????????????????????????????????????????????
              12900 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562780 ????????????????????????????????????????????????????????????
              12960 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562840 ????????????????????????????????????????????????????????????
              13020 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562900 ????????????????????????????????????????????????????????????
              13080 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75562960 ????????????????????????????????????????????????????????????
              13140 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563020 ????????????????????????????????????????????????????????????
              13200 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563080 ????????????????????????????????????????????????????????????
              13260 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563140 ????????????????????????????????????????????????????????????
              13320 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563200 ????????????????????????????????????????????????????????????
              13380 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563260 ????????????????????????????????????????????????????????????
              13440 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563320 ????????????????????????????????????????????????????????????
              13500 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563380 ????????????????????????????????????????????????????????????
              13560 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563440 ????????????????????????????????????????????????????????????
              13620 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563500 ????????????????????????????????????????????????????????????
              13680 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563560 ????????????????????????????????????????????????????????????
              13740 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563620 ????????????????????????????????????????????????????????????
              13800 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563680 ????????????????????????????????????????????????????????????
              13860 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563740 ????????????????????????????????????????????????????????????
              13920 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563800 ????????????????????????????????????????????????????????????
              13980 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563860 ????????????????????????????????????????????????????????????
              14040 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563920 ????????????????????????????????????????????????????????????
              14100 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75563980 ????????????????????????????????????????????????????????????
              14160 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564040 ????????????????????????????????????????????????????????????
              14220 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564100 ????????????????????????????????????????????????????????????
              14280 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564160 ????????????????????????????????????????????????????????????
              14340 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564220 ????????????????????????????????????????????????????????????
              14400 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564280 ????????????????????????????????????????????????????????????
              14460 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564340 ????????????????????????????????????????????????????????????
              14520 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564400 ????????????????????????????????????????????????????????????
              14580 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564460 ????????????????????????????????????????????????????????????
              14640 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564520 ????????????????????????????????????????????????????????????
              14700 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564580 ????????????????????????????????????????????????????????????
              14760 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564640 ????????????????????????????????????????????????????????????
              14820 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564700 ????????????????????????????????????????????????????????????
              14880 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564760 ????????????????????????????????????????????????????????????
              14940 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564820 ????????????????????????????????????????????????????????????
              15000 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564880 ????????????????????????????????????????????????????????????
              15060 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75564940 ????????????????????????????????????????????????????????????
              15120 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565000 ????????????????????????????????????????????????????????????
              15180 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565060 ????????????????????????????????????????????????????????????
              15240 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565120 ????????????????????????????????????????????????????????????
              15300 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565180 ????????????????????????????????????????????????????????????
              15360 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565240 ????????????????????????????????????????????????????????????
              15420 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565300 ????????????????????????????????????????????????????????????
              15480 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565360 ????????????????????????????????????????????????????????????
              15540 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565420 ????????????????????????????????????????????????????????????
              15600 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565480 ????????????????????????????????????????????????????????????
              15660 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565540 ????????????????????????????????????????????????????????????
              15720 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565600 ????????????????????????????????????????????????????????????
              15780 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565660 ????????????????????????????????????????????????????????????
              15840 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565720 ????????????????????????????????????????????????????????????
              15900 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565780 ????????????????????????????????????????????????????????????
              15960 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565840 ????????????????????????????????????????????????????????????
              16020 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565900 ????????????????????????????????????????????????????????????
              16080 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75565960 ????????????????????????????????????????????????????????????
              16140 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566020 ????????????????????????????????????????????????????????????
              16200 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566080 ????????????????????????????????????????????????????????????
              16260 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566140 ????????????????????????????????????????????????????????????
              16320 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566200 ????????????????????????????????????????????????????????????
              16380 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566260 ????????????????????????????????????????????????????????????
              16440 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566320 ????????????????????????????????????????????????????????????
              16500 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566380 ????????????????????????????????????????????????????????????
              16560 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566440 ????????????????????????????????????????????????????????????
              16620 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566500 ????????????????????????????????????????????????????????????
              16680 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566560 ????????????????????????????????????????????????????????????
              16740 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566620 ????????????????????????????????????????????????????????????
              16800 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566680 ????????????????????????????????????????????????????????????
              16860 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566740 ????????????????????????????????????????????????????????????
              16920 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566800 ????????????????????????????????????????????????????????????
              16980 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566860 ????????????????????????????????????????????????????????????
              17040 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566920 ????????????????????????????????????????????????????????????
              17100 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75566980 ????????????????????????????????????????????????????????????
              17160 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75567040 ????????????????????????????????????????????????????????????
              17220 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75567100 ????????????????????????????????????????????????????????????
              17280 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75567160 ????????????????????????????????????????????????????????????
              17340 ------------------------------------------------------------
CAG33136.        45 ------------------------------------------------------------

chr13      75567220 ????????????????????????????????????????????????????????????
              17400 -----|||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.        45 -----???????????????????????????????????????????????????????

chr13      75567280 ???????????????????????????????? 75567312
              17460 ||||||||||||||||||||||||||||||||    17492
CAG33136.       100 ????????????????????????????????      132
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	75549820	75567312	CAG33136.1	986	+	75549820	75567312	0	2	45,87,	0,17405,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (132 aligned letters; 0 identities; 0 mismatches; 17360 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 132:
        identities = 0,
        mismatches = 0.
    gaps = 17360:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 17360:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 17360:
                open_internal_deletions = 1,
                extend_internal_deletions = 17359;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 17360)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 17360)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 17360)
        self.assertEqual(counts.gaps, 17360)
        self.assertEqual(counts.aligned, 132)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[75604767, 75604827, 75605728, 75605809],
                          [       0,       60,       60,      141]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      75604767 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.         0 ????????????????????????????????????????????????????????????

chr13      75604827 ????????????????????????????????????????????????????????????
                 60 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75604887 ????????????????????????????????????????????????????????????
                120 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75604947 ????????????????????????????????????????????????????????????
                180 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605007 ????????????????????????????????????????????????????????????
                240 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605067 ????????????????????????????????????????????????????????????
                300 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605127 ????????????????????????????????????????????????????????????
                360 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605187 ????????????????????????????????????????????????????????????
                420 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605247 ????????????????????????????????????????????????????????????
                480 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605307 ????????????????????????????????????????????????????????????
                540 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605367 ????????????????????????????????????????????????????????????
                600 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605427 ????????????????????????????????????????????????????????????
                660 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605487 ????????????????????????????????????????????????????????????
                720 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605547 ????????????????????????????????????????????????????????????
                780 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605607 ????????????????????????????????????????????????????????????
                840 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605667 ????????????????????????????????????????????????????????????
                900 ------------------------------------------------------------
CAG33136.        60 ------------------------------------------------------------

chr13      75605727 ????????????????????????????????????????????????????????????
                960 -|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.        60 -???????????????????????????????????????????????????????????

chr13      75605787 ?????????????????????? 75605809
               1020 ||||||||||||||||||||||     1042
CAG33136.       119 ??????????????????????      141
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	75604767	75605809	CAG33136.1	1000	+	75604767	75605809	0	2	60,81,	0,961,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (141 aligned letters; 0 identities; 0 mismatches; 901 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 141:
        identities = 0,
        mismatches = 0.
    gaps = 901:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 901:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 901:
                open_internal_deletions = 1,
                extend_internal_deletions = 900;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 901)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 901)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 901)
        self.assertEqual(counts.gaps, 901)
        self.assertEqual(counts.aligned, 141)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[75594914, 75594989],
                          [       0,       75]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      75594914 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.         0 ????????????????????????????????????????????????????????????

chr13      75594974 ??????????????? 75594989
                 60 |||||||||||||||       75
CAG33136.        60 ???????????????       75
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	75594914	75594989	CAG33136.1	1000	+	75594914	75594989	0	1	75,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (75 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 75:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 75)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1000)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr13")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[75569459, 75569507],
                          [       0,       48]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr13      75569459 ???????????????????????????????????????????????? 75569507
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||       48
CAG33136.         0 ????????????????????????????????????????????????       48
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr13	75569459	75569507	CAG33136.1	1000	+	75569459	75569507	0	1	48,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (48 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 48:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 48)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 530)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr4")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[41260685, 41260787],
                          [       0,      102]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr4       41260685 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.         0 ????????????????????????????????????????????????????????????

chr4       41260745 ?????????????????????????????????????????? 41260787
                 60 ||||||||||||||||||||||||||||||||||||||||||      102
CAG33136.        60 ??????????????????????????????????????????      102
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr4	41260685	41260787	CAG33136.1	530	+	41260685	41260787	0	1	102,	0,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (102 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 102:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 102)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 166)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr4")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[41257605, 41257731, 41263227, 41263290],
                          [       0,      126,      126,      189]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
chr4       41257605 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.         0 ????????????????????????????????????????????????????????????

chr4       41257665 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.        60 ????????????????????????????????????????????????????????????

chr4       41257725 ????????????????????????????????????????????????????????????
                120 ||||||------------------------------------------------------
CAG33136.       120 ??????------------------------------------------------------

chr4       41257785 ????????????????????????????????????????????????????????????
                180 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41257845 ????????????????????????????????????????????????????????????
                240 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41257905 ????????????????????????????????????????????????????????????
                300 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41257965 ????????????????????????????????????????????????????????????
                360 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258025 ????????????????????????????????????????????????????????????
                420 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258085 ????????????????????????????????????????????????????????????
                480 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258145 ????????????????????????????????????????????????????????????
                540 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258205 ????????????????????????????????????????????????????????????
                600 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258265 ????????????????????????????????????????????????????????????
                660 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258325 ????????????????????????????????????????????????????????????
                720 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258385 ????????????????????????????????????????????????????????????
                780 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258445 ????????????????????????????????????????????????????????????
                840 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258505 ????????????????????????????????????????????????????????????
                900 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258565 ????????????????????????????????????????????????????????????
                960 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258625 ????????????????????????????????????????????????????????????
               1020 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258685 ????????????????????????????????????????????????????????????
               1080 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258745 ????????????????????????????????????????????????????????????
               1140 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258805 ????????????????????????????????????????????????????????????
               1200 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258865 ????????????????????????????????????????????????????????????
               1260 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258925 ????????????????????????????????????????????????????????????
               1320 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41258985 ????????????????????????????????????????????????????????????
               1380 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259045 ????????????????????????????????????????????????????????????
               1440 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259105 ????????????????????????????????????????????????????????????
               1500 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259165 ????????????????????????????????????????????????????????????
               1560 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259225 ????????????????????????????????????????????????????????????
               1620 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259285 ????????????????????????????????????????????????????????????
               1680 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259345 ????????????????????????????????????????????????????????????
               1740 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259405 ????????????????????????????????????????????????????????????
               1800 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259465 ????????????????????????????????????????????????????????????
               1860 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259525 ????????????????????????????????????????????????????????????
               1920 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259585 ????????????????????????????????????????????????????????????
               1980 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259645 ????????????????????????????????????????????????????????????
               2040 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259705 ????????????????????????????????????????????????????????????
               2100 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259765 ????????????????????????????????????????????????????????????
               2160 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259825 ????????????????????????????????????????????????????????????
               2220 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259885 ????????????????????????????????????????????????????????????
               2280 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41259945 ????????????????????????????????????????????????????????????
               2340 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260005 ????????????????????????????????????????????????????????????
               2400 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260065 ????????????????????????????????????????????????????????????
               2460 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260125 ????????????????????????????????????????????????????????????
               2520 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260185 ????????????????????????????????????????????????????????????
               2580 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260245 ????????????????????????????????????????????????????????????
               2640 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260305 ????????????????????????????????????????????????????????????
               2700 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260365 ????????????????????????????????????????????????????????????
               2760 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260425 ????????????????????????????????????????????????????????????
               2820 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260485 ????????????????????????????????????????????????????????????
               2880 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260545 ????????????????????????????????????????????????????????????
               2940 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260605 ????????????????????????????????????????????????????????????
               3000 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260665 ????????????????????????????????????????????????????????????
               3060 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260725 ????????????????????????????????????????????????????????????
               3120 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260785 ????????????????????????????????????????????????????????????
               3180 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260845 ????????????????????????????????????????????????????????????
               3240 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260905 ????????????????????????????????????????????????????????????
               3300 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41260965 ????????????????????????????????????????????????????????????
               3360 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261025 ????????????????????????????????????????????????????????????
               3420 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261085 ????????????????????????????????????????????????????????????
               3480 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261145 ????????????????????????????????????????????????????????????
               3540 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261205 ????????????????????????????????????????????????????????????
               3600 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261265 ????????????????????????????????????????????????????????????
               3660 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261325 ????????????????????????????????????????????????????????????
               3720 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261385 ????????????????????????????????????????????????????????????
               3780 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261445 ????????????????????????????????????????????????????????????
               3840 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261505 ????????????????????????????????????????????????????????????
               3900 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261565 ????????????????????????????????????????????????????????????
               3960 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261625 ????????????????????????????????????????????????????????????
               4020 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261685 ????????????????????????????????????????????????????????????
               4080 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261745 ????????????????????????????????????????????????????????????
               4140 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261805 ????????????????????????????????????????????????????????????
               4200 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261865 ????????????????????????????????????????????????????????????
               4260 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261925 ????????????????????????????????????????????????????????????
               4320 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41261985 ????????????????????????????????????????????????????????????
               4380 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262045 ????????????????????????????????????????????????????????????
               4440 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262105 ????????????????????????????????????????????????????????????
               4500 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262165 ????????????????????????????????????????????????????????????
               4560 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262225 ????????????????????????????????????????????????????????????
               4620 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262285 ????????????????????????????????????????????????????????????
               4680 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262345 ????????????????????????????????????????????????????????????
               4740 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262405 ????????????????????????????????????????????????????????????
               4800 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262465 ????????????????????????????????????????????????????????????
               4860 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262525 ????????????????????????????????????????????????????????????
               4920 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262585 ????????????????????????????????????????????????????????????
               4980 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262645 ????????????????????????????????????????????????????????????
               5040 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262705 ????????????????????????????????????????????????????????????
               5100 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262765 ????????????????????????????????????????????????????????????
               5160 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262825 ????????????????????????????????????????????????????????????
               5220 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262885 ????????????????????????????????????????????????????????????
               5280 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41262945 ????????????????????????????????????????????????????????????
               5340 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41263005 ????????????????????????????????????????????????????????????
               5400 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41263065 ????????????????????????????????????????????????????????????
               5460 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41263125 ????????????????????????????????????????????????????????????
               5520 ------------------------------------------------------------
CAG33136.       126 ------------------------------------------------------------

chr4       41263185 ????????????????????????????????????????????????????????????
               5580 ------------------------------------------||||||||||||||||||
CAG33136.       126 ------------------------------------------??????????????????

chr4       41263245 ????????????????????????????????????????????? 41263290
               5640 |||||||||||||||||||||||||||||||||||||||||||||     5685
CAG33136.       144 ?????????????????????????????????????????????      189
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
chr4	41257605	41263290	CAG33136.1	166	+	41257605	41263290	0	2	126,63,	0,5622,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (189 aligned letters; 0 identities; 0 mismatches; 5496 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 189:
        identities = 0,
        mismatches = 0.
    gaps = 5496:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 5496:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 5496:
                open_internal_deletions = 1,
                extend_internal_deletions = 5495;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 5496)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 5496)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 5496)
        self.assertEqual(counts.gaps, 5496)
        self.assertEqual(counts.aligned, 189)
        self.assertRaises(StopIteration, next, alignments)

    def test_writing_psl_35_001(self):
        """Test writing the alignments in psl_35_001.bed."""
        path = "Blat/psl_35_001.bed"
        with open(path) as stream:
            original_data = stream.read()
        alignments = Align.parse(path, "bed")
        stream = StringIO()
        n = Align.write(alignments, stream, "bed")
        self.assertEqual(n, 8)
        stream.seek(0)
        written_data = stream.read()
        stream.close()
        self.assertEqual(original_data, written_data)

    def test_reading_psl_35_002(self):
        """Test parsing psl_35_002.bed."""
        path = "Blat/psl_35_002.bed"
        alignments = Align.parse(path, "bed")
        alignment = next(alignments)
        self.assertEqual(alignment.score, 972)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "KI537979")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[9712654, 9712786, 9715941, 9716097, 9716445, 9716532, 9718374,
                           9718422, 9739264, 9739339, 9743706, 9743766, 9744511, 9744592],
                          [      0,     132,     132,     288,     288,     375,     375,
                               423,     423,     498,     498,     558,     558,     639]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
KI537979    9712654 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.         0 ????????????????????????????????????????????????????????????

KI537979    9712714 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.        60 ????????????????????????????????????????????????????????????

KI537979    9712774 ????????????????????????????????????????????????????????????
                120 ||||||||||||------------------------------------------------
CAG33136.       120 ????????????------------------------------------------------

KI537979    9712834 ????????????????????????????????????????????????????????????
                180 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9712894 ????????????????????????????????????????????????????????????
                240 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9712954 ????????????????????????????????????????????????????????????
                300 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713014 ????????????????????????????????????????????????????????????
                360 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713074 ????????????????????????????????????????????????????????????
                420 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713134 ????????????????????????????????????????????????????????????
                480 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713194 ????????????????????????????????????????????????????????????
                540 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713254 ????????????????????????????????????????????????????????????
                600 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713314 ????????????????????????????????????????????????????????????
                660 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713374 ????????????????????????????????????????????????????????????
                720 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713434 ????????????????????????????????????????????????????????????
                780 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713494 ????????????????????????????????????????????????????????????
                840 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713554 ????????????????????????????????????????????????????????????
                900 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713614 ????????????????????????????????????????????????????????????
                960 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713674 ????????????????????????????????????????????????????????????
               1020 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713734 ????????????????????????????????????????????????????????????
               1080 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713794 ????????????????????????????????????????????????????????????
               1140 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713854 ????????????????????????????????????????????????????????????
               1200 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713914 ????????????????????????????????????????????????????????????
               1260 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9713974 ????????????????????????????????????????????????????????????
               1320 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714034 ????????????????????????????????????????????????????????????
               1380 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714094 ????????????????????????????????????????????????????????????
               1440 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714154 ????????????????????????????????????????????????????????????
               1500 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714214 ????????????????????????????????????????????????????????????
               1560 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714274 ????????????????????????????????????????????????????????????
               1620 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714334 ????????????????????????????????????????????????????????????
               1680 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714394 ????????????????????????????????????????????????????????????
               1740 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714454 ????????????????????????????????????????????????????????????
               1800 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714514 ????????????????????????????????????????????????????????????
               1860 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714574 ????????????????????????????????????????????????????????????
               1920 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714634 ????????????????????????????????????????????????????????????
               1980 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714694 ????????????????????????????????????????????????????????????
               2040 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714754 ????????????????????????????????????????????????????????????
               2100 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714814 ????????????????????????????????????????????????????????????
               2160 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714874 ????????????????????????????????????????????????????????????
               2220 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714934 ????????????????????????????????????????????????????????????
               2280 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9714994 ????????????????????????????????????????????????????????????
               2340 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715054 ????????????????????????????????????????????????????????????
               2400 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715114 ????????????????????????????????????????????????????????????
               2460 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715174 ????????????????????????????????????????????????????????????
               2520 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715234 ????????????????????????????????????????????????????????????
               2580 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715294 ????????????????????????????????????????????????????????????
               2640 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715354 ????????????????????????????????????????????????????????????
               2700 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715414 ????????????????????????????????????????????????????????????
               2760 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715474 ????????????????????????????????????????????????????????????
               2820 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715534 ????????????????????????????????????????????????????????????
               2880 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715594 ????????????????????????????????????????????????????????????
               2940 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715654 ????????????????????????????????????????????????????????????
               3000 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715714 ????????????????????????????????????????????????????????????
               3060 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715774 ????????????????????????????????????????????????????????????
               3120 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715834 ????????????????????????????????????????????????????????????
               3180 ------------------------------------------------------------
CAG33136.       132 ------------------------------------------------------------

KI537979    9715894 ????????????????????????????????????????????????????????????
               3240 -----------------------------------------------|||||||||||||
CAG33136.       132 -----------------------------------------------?????????????

KI537979    9715954 ????????????????????????????????????????????????????????????
               3300 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       145 ????????????????????????????????????????????????????????????

KI537979    9716014 ????????????????????????????????????????????????????????????
               3360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       205 ????????????????????????????????????????????????????????????

KI537979    9716074 ????????????????????????????????????????????????????????????
               3420 |||||||||||||||||||||||-------------------------------------
CAG33136.       265 ???????????????????????-------------------------------------

KI537979    9716134 ????????????????????????????????????????????????????????????
               3480 ------------------------------------------------------------
CAG33136.       288 ------------------------------------------------------------

KI537979    9716194 ????????????????????????????????????????????????????????????
               3540 ------------------------------------------------------------
CAG33136.       288 ------------------------------------------------------------

KI537979    9716254 ????????????????????????????????????????????????????????????
               3600 ------------------------------------------------------------
CAG33136.       288 ------------------------------------------------------------

KI537979    9716314 ????????????????????????????????????????????????????????????
               3660 ------------------------------------------------------------
CAG33136.       288 ------------------------------------------------------------

KI537979    9716374 ????????????????????????????????????????????????????????????
               3720 ------------------------------------------------------------
CAG33136.       288 ------------------------------------------------------------

KI537979    9716434 ????????????????????????????????????????????????????????????
               3780 -----------|||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       288 -----------?????????????????????????????????????????????????

KI537979    9716494 ????????????????????????????????????????????????????????????
               3840 ||||||||||||||||||||||||||||||||||||||----------------------
CAG33136.       337 ??????????????????????????????????????----------------------

KI537979    9716554 ????????????????????????????????????????????????????????????
               3900 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9716614 ????????????????????????????????????????????????????????????
               3960 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9716674 ????????????????????????????????????????????????????????????
               4020 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9716734 ????????????????????????????????????????????????????????????
               4080 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9716794 ????????????????????????????????????????????????????????????
               4140 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9716854 ????????????????????????????????????????????????????????????
               4200 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9716914 ????????????????????????????????????????????????????????????
               4260 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9716974 ????????????????????????????????????????????????????????????
               4320 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717034 ????????????????????????????????????????????????????????????
               4380 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717094 ????????????????????????????????????????????????????????????
               4440 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717154 ????????????????????????????????????????????????????????????
               4500 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717214 ????????????????????????????????????????????????????????????
               4560 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717274 ????????????????????????????????????????????????????????????
               4620 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717334 ????????????????????????????????????????????????????????????
               4680 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717394 ????????????????????????????????????????????????????????????
               4740 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717454 ????????????????????????????????????????????????????????????
               4800 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717514 ????????????????????????????????????????????????????????????
               4860 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717574 ????????????????????????????????????????????????????????????
               4920 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717634 ????????????????????????????????????????????????????????????
               4980 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717694 ????????????????????????????????????????????????????????????
               5040 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717754 ????????????????????????????????????????????????????????????
               5100 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717814 ????????????????????????????????????????????????????????????
               5160 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717874 ????????????????????????????????????????????????????????????
               5220 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717934 ????????????????????????????????????????????????????????????
               5280 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9717994 ????????????????????????????????????????????????????????????
               5340 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9718054 ????????????????????????????????????????????????????????????
               5400 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9718114 ????????????????????????????????????????????????????????????
               5460 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9718174 ????????????????????????????????????????????????????????????
               5520 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9718234 ????????????????????????????????????????????????????????????
               5580 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9718294 ????????????????????????????????????????????????????????????
               5640 ------------------------------------------------------------
CAG33136.       375 ------------------------------------------------------------

KI537979    9718354 ????????????????????????????????????????????????????????????
               5700 --------------------||||||||||||||||||||||||||||||||||||||||
CAG33136.       375 --------------------????????????????????????????????????????

KI537979    9718414 ????????????????????????????????????????????????????????????
               5760 ||||||||----------------------------------------------------
CAG33136.       415 ????????----------------------------------------------------

KI537979    9718474 ????????????????????????????????????????????????????????????
               5820 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9718534 ????????????????????????????????????????????????????????????
               5880 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9718594 ????????????????????????????????????????????????????????????
               5940 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9718654 ????????????????????????????????????????????????????????????
               6000 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9718714 ????????????????????????????????????????????????????????????
               6060 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9718774 ????????????????????????????????????????????????????????????
               6120 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9718834 ????????????????????????????????????????????????????????????
               6180 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9718894 ????????????????????????????????????????????????????????????
               6240 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9718954 ????????????????????????????????????????????????????????????
               6300 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719014 ????????????????????????????????????????????????????????????
               6360 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719074 ????????????????????????????????????????????????????????????
               6420 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719134 ????????????????????????????????????????????????????????????
               6480 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719194 ????????????????????????????????????????????????????????????
               6540 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719254 ????????????????????????????????????????????????????????????
               6600 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719314 ????????????????????????????????????????????????????????????
               6660 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719374 ????????????????????????????????????????????????????????????
               6720 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719434 ????????????????????????????????????????????????????????????
               6780 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719494 ????????????????????????????????????????????????????????????
               6840 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719554 ????????????????????????????????????????????????????????????
               6900 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719614 ????????????????????????????????????????????????????????????
               6960 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719674 ????????????????????????????????????????????????????????????
               7020 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719734 ????????????????????????????????????????????????????????????
               7080 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719794 ????????????????????????????????????????????????????????????
               7140 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719854 ????????????????????????????????????????????????????????????
               7200 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719914 ????????????????????????????????????????????????????????????
               7260 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9719974 ????????????????????????????????????????????????????????????
               7320 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720034 ????????????????????????????????????????????????????????????
               7380 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720094 ????????????????????????????????????????????????????????????
               7440 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720154 ????????????????????????????????????????????????????????????
               7500 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720214 ????????????????????????????????????????????????????????????
               7560 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720274 ????????????????????????????????????????????????????????????
               7620 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720334 ????????????????????????????????????????????????????????????
               7680 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720394 ????????????????????????????????????????????????????????????
               7740 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720454 ????????????????????????????????????????????????????????????
               7800 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720514 ????????????????????????????????????????????????????????????
               7860 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720574 ????????????????????????????????????????????????????????????
               7920 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720634 ????????????????????????????????????????????????????????????
               7980 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720694 ????????????????????????????????????????????????????????????
               8040 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720754 ????????????????????????????????????????????????????????????
               8100 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720814 ????????????????????????????????????????????????????????????
               8160 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720874 ????????????????????????????????????????????????????????????
               8220 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720934 ????????????????????????????????????????????????????????????
               8280 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9720994 ????????????????????????????????????????????????????????????
               8340 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721054 ????????????????????????????????????????????????????????????
               8400 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721114 ????????????????????????????????????????????????????????????
               8460 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721174 ????????????????????????????????????????????????????????????
               8520 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721234 ????????????????????????????????????????????????????????????
               8580 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721294 ????????????????????????????????????????????????????????????
               8640 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721354 ????????????????????????????????????????????????????????????
               8700 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721414 ????????????????????????????????????????????????????????????
               8760 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721474 ????????????????????????????????????????????????????????????
               8820 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721534 ????????????????????????????????????????????????????????????
               8880 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721594 ????????????????????????????????????????????????????????????
               8940 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721654 ????????????????????????????????????????????????????????????
               9000 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721714 ????????????????????????????????????????????????????????????
               9060 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721774 ????????????????????????????????????????????????????????????
               9120 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721834 ????????????????????????????????????????????????????????????
               9180 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721894 ????????????????????????????????????????????????????????????
               9240 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9721954 ????????????????????????????????????????????????????????????
               9300 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722014 ????????????????????????????????????????????????????????????
               9360 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722074 ????????????????????????????????????????????????????????????
               9420 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722134 ????????????????????????????????????????????????????????????
               9480 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722194 ????????????????????????????????????????????????????????????
               9540 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722254 ????????????????????????????????????????????????????????????
               9600 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722314 ????????????????????????????????????????????????????????????
               9660 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722374 ????????????????????????????????????????????????????????????
               9720 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722434 ????????????????????????????????????????????????????????????
               9780 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722494 ????????????????????????????????????????????????????????????
               9840 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722554 ????????????????????????????????????????????????????????????
               9900 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722614 ????????????????????????????????????????????????????????????
               9960 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722674 ????????????????????????????????????????????????????????????
              10020 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722734 ????????????????????????????????????????????????????????????
              10080 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722794 ????????????????????????????????????????????????????????????
              10140 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722854 ????????????????????????????????????????????????????????????
              10200 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722914 ????????????????????????????????????????????????????????????
              10260 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9722974 ????????????????????????????????????????????????????????????
              10320 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723034 ????????????????????????????????????????????????????????????
              10380 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723094 ????????????????????????????????????????????????????????????
              10440 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723154 ????????????????????????????????????????????????????????????
              10500 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723214 ????????????????????????????????????????????????????????????
              10560 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723274 ????????????????????????????????????????????????????????????
              10620 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723334 ????????????????????????????????????????????????????????????
              10680 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723394 ????????????????????????????????????????????????????????????
              10740 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723454 ????????????????????????????????????????????????????????????
              10800 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723514 ????????????????????????????????????????????????????????????
              10860 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723574 ????????????????????????????????????????????????????????????
              10920 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723634 ????????????????????????????????????????????????????????????
              10980 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723694 ????????????????????????????????????????????????????????????
              11040 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723754 ????????????????????????????????????????????????????????????
              11100 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723814 ????????????????????????????????????????????????????????????
              11160 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723874 ????????????????????????????????????????????????????????????
              11220 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723934 ????????????????????????????????????????????????????????????
              11280 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9723994 ????????????????????????????????????????????????????????????
              11340 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724054 ????????????????????????????????????????????????????????????
              11400 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724114 ????????????????????????????????????????????????????????????
              11460 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724174 ????????????????????????????????????????????????????????????
              11520 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724234 ????????????????????????????????????????????????????????????
              11580 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724294 ????????????????????????????????????????????????????????????
              11640 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724354 ????????????????????????????????????????????????????????????
              11700 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724414 ????????????????????????????????????????????????????????????
              11760 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724474 ????????????????????????????????????????????????????????????
              11820 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724534 ????????????????????????????????????????????????????????????
              11880 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724594 ????????????????????????????????????????????????????????????
              11940 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724654 ????????????????????????????????????????????????????????????
              12000 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724714 ????????????????????????????????????????????????????????????
              12060 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724774 ????????????????????????????????????????????????????????????
              12120 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724834 ????????????????????????????????????????????????????????????
              12180 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724894 ????????????????????????????????????????????????????????????
              12240 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9724954 ????????????????????????????????????????????????????????????
              12300 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725014 ????????????????????????????????????????????????????????????
              12360 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725074 ????????????????????????????????????????????????????????????
              12420 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725134 ????????????????????????????????????????????????????????????
              12480 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725194 ????????????????????????????????????????????????????????????
              12540 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725254 ????????????????????????????????????????????????????????????
              12600 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725314 ????????????????????????????????????????????????????????????
              12660 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725374 ????????????????????????????????????????????????????????????
              12720 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725434 ????????????????????????????????????????????????????????????
              12780 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725494 ????????????????????????????????????????????????????????????
              12840 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725554 ????????????????????????????????????????????????????????????
              12900 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725614 ????????????????????????????????????????????????????????????
              12960 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725674 ????????????????????????????????????????????????????????????
              13020 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725734 ????????????????????????????????????????????????????????????
              13080 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725794 ????????????????????????????????????????????????????????????
              13140 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725854 ????????????????????????????????????????????????????????????
              13200 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725914 ????????????????????????????????????????????????????????????
              13260 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9725974 ????????????????????????????????????????????????????????????
              13320 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726034 ????????????????????????????????????????????????????????????
              13380 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726094 ????????????????????????????????????????????????????????????
              13440 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726154 ????????????????????????????????????????????????????????????
              13500 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726214 ????????????????????????????????????????????????????????????
              13560 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726274 ????????????????????????????????????????????????????????????
              13620 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726334 ????????????????????????????????????????????????????????????
              13680 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726394 ????????????????????????????????????????????????????????????
              13740 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726454 ????????????????????????????????????????????????????????????
              13800 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726514 ????????????????????????????????????????????????????????????
              13860 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726574 ????????????????????????????????????????????????????????????
              13920 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726634 ????????????????????????????????????????????????????????????
              13980 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726694 ????????????????????????????????????????????????????????????
              14040 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726754 ????????????????????????????????????????????????????????????
              14100 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726814 ????????????????????????????????????????????????????????????
              14160 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726874 ????????????????????????????????????????????????????????????
              14220 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726934 ????????????????????????????????????????????????????????????
              14280 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9726994 ????????????????????????????????????????????????????????????
              14340 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727054 ????????????????????????????????????????????????????????????
              14400 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727114 ????????????????????????????????????????????????????????????
              14460 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727174 ????????????????????????????????????????????????????????????
              14520 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727234 ????????????????????????????????????????????????????????????
              14580 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727294 ????????????????????????????????????????????????????????????
              14640 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727354 ????????????????????????????????????????????????????????????
              14700 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727414 ????????????????????????????????????????????????????????????
              14760 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727474 ????????????????????????????????????????????????????????????
              14820 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727534 ????????????????????????????????????????????????????????????
              14880 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727594 ????????????????????????????????????????????????????????????
              14940 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727654 ????????????????????????????????????????????????????????????
              15000 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727714 ????????????????????????????????????????????????????????????
              15060 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727774 ????????????????????????????????????????????????????????????
              15120 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727834 ????????????????????????????????????????????????????????????
              15180 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727894 ????????????????????????????????????????????????????????????
              15240 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9727954 ????????????????????????????????????????????????????????????
              15300 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728014 ????????????????????????????????????????????????????????????
              15360 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728074 ????????????????????????????????????????????????????????????
              15420 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728134 ????????????????????????????????????????????????????????????
              15480 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728194 ????????????????????????????????????????????????????????????
              15540 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728254 ????????????????????????????????????????????????????????????
              15600 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728314 ????????????????????????????????????????????????????????????
              15660 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728374 ????????????????????????????????????????????????????????????
              15720 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728434 ????????????????????????????????????????????????????????????
              15780 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728494 ????????????????????????????????????????????????????????????
              15840 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728554 ????????????????????????????????????????????????????????????
              15900 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728614 ????????????????????????????????????????????????????????????
              15960 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728674 ????????????????????????????????????????????????????????????
              16020 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728734 ????????????????????????????????????????????????????????????
              16080 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728794 ????????????????????????????????????????????????????????????
              16140 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728854 ????????????????????????????????????????????????????????????
              16200 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728914 ????????????????????????????????????????????????????????????
              16260 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9728974 ????????????????????????????????????????????????????????????
              16320 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729034 ????????????????????????????????????????????????????????????
              16380 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729094 ????????????????????????????????????????????????????????????
              16440 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729154 ????????????????????????????????????????????????????????????
              16500 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729214 ????????????????????????????????????????????????????????????
              16560 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729274 ????????????????????????????????????????????????????????????
              16620 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729334 ????????????????????????????????????????????????????????????
              16680 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729394 ????????????????????????????????????????????????????????????
              16740 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729454 ????????????????????????????????????????????????????????????
              16800 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729514 ????????????????????????????????????????????????????????????
              16860 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729574 ????????????????????????????????????????????????????????????
              16920 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729634 ????????????????????????????????????????????????????????????
              16980 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729694 ????????????????????????????????????????????????????????????
              17040 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729754 ????????????????????????????????????????????????????????????
              17100 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729814 ????????????????????????????????????????????????????????????
              17160 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729874 ????????????????????????????????????????????????????????????
              17220 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729934 ????????????????????????????????????????????????????????????
              17280 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9729994 ????????????????????????????????????????????????????????????
              17340 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730054 ????????????????????????????????????????????????????????????
              17400 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730114 ????????????????????????????????????????????????????????????
              17460 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730174 ????????????????????????????????????????????????????????????
              17520 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730234 ????????????????????????????????????????????????????????????
              17580 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730294 ????????????????????????????????????????????????????????????
              17640 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730354 ????????????????????????????????????????????????????????????
              17700 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730414 ????????????????????????????????????????????????????????????
              17760 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730474 ????????????????????????????????????????????????????????????
              17820 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730534 ????????????????????????????????????????????????????????????
              17880 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730594 ????????????????????????????????????????????????????????????
              17940 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730654 ????????????????????????????????????????????????????????????
              18000 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730714 ????????????????????????????????????????????????????????????
              18060 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730774 ????????????????????????????????????????????????????????????
              18120 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730834 ????????????????????????????????????????????????????????????
              18180 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730894 ????????????????????????????????????????????????????????????
              18240 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9730954 ????????????????????????????????????????????????????????????
              18300 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731014 ????????????????????????????????????????????????????????????
              18360 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731074 ????????????????????????????????????????????????????????????
              18420 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731134 ????????????????????????????????????????????????????????????
              18480 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731194 ????????????????????????????????????????????????????????????
              18540 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731254 ????????????????????????????????????????????????????????????
              18600 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731314 ????????????????????????????????????????????????????????????
              18660 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731374 ????????????????????????????????????????????????????????????
              18720 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731434 ????????????????????????????????????????????????????????????
              18780 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731494 ????????????????????????????????????????????????????????????
              18840 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731554 ????????????????????????????????????????????????????????????
              18900 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731614 ????????????????????????????????????????????????????????????
              18960 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731674 ????????????????????????????????????????????????????????????
              19020 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731734 ????????????????????????????????????????????????????????????
              19080 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731794 ????????????????????????????????????????????????????????????
              19140 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731854 ????????????????????????????????????????????????????????????
              19200 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731914 ????????????????????????????????????????????????????????????
              19260 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9731974 ????????????????????????????????????????????????????????????
              19320 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732034 ????????????????????????????????????????????????????????????
              19380 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732094 ????????????????????????????????????????????????????????????
              19440 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732154 ????????????????????????????????????????????????????????????
              19500 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732214 ????????????????????????????????????????????????????????????
              19560 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732274 ????????????????????????????????????????????????????????????
              19620 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732334 ????????????????????????????????????????????????????????????
              19680 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732394 ????????????????????????????????????????????????????????????
              19740 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732454 ????????????????????????????????????????????????????????????
              19800 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732514 ????????????????????????????????????????????????????????????
              19860 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732574 ????????????????????????????????????????????????????????????
              19920 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732634 ????????????????????????????????????????????????????????????
              19980 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732694 ????????????????????????????????????????????????????????????
              20040 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732754 ????????????????????????????????????????????????????????????
              20100 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732814 ????????????????????????????????????????????????????????????
              20160 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732874 ????????????????????????????????????????????????????????????
              20220 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732934 ????????????????????????????????????????????????????????????
              20280 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9732994 ????????????????????????????????????????????????????????????
              20340 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733054 ????????????????????????????????????????????????????????????
              20400 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733114 ????????????????????????????????????????????????????????????
              20460 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733174 ????????????????????????????????????????????????????????????
              20520 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733234 ????????????????????????????????????????????????????????????
              20580 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733294 ????????????????????????????????????????????????????????????
              20640 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733354 ????????????????????????????????????????????????????????????
              20700 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733414 ????????????????????????????????????????????????????????????
              20760 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733474 ????????????????????????????????????????????????????????????
              20820 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733534 ????????????????????????????????????????????????????????????
              20880 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733594 ????????????????????????????????????????????????????????????
              20940 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733654 ????????????????????????????????????????????????????????????
              21000 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733714 ????????????????????????????????????????????????????????????
              21060 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733774 ????????????????????????????????????????????????????????????
              21120 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733834 ????????????????????????????????????????????????????????????
              21180 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733894 ????????????????????????????????????????????????????????????
              21240 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9733954 ????????????????????????????????????????????????????????????
              21300 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734014 ????????????????????????????????????????????????????????????
              21360 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734074 ????????????????????????????????????????????????????????????
              21420 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734134 ????????????????????????????????????????????????????????????
              21480 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734194 ????????????????????????????????????????????????????????????
              21540 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734254 ????????????????????????????????????????????????????????????
              21600 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734314 ????????????????????????????????????????????????????????????
              21660 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734374 ????????????????????????????????????????????????????????????
              21720 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734434 ????????????????????????????????????????????????????????????
              21780 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734494 ????????????????????????????????????????????????????????????
              21840 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734554 ????????????????????????????????????????????????????????????
              21900 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734614 ????????????????????????????????????????????????????????????
              21960 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734674 ????????????????????????????????????????????????????????????
              22020 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734734 ????????????????????????????????????????????????????????????
              22080 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734794 ????????????????????????????????????????????????????????????
              22140 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734854 ????????????????????????????????????????????????????????????
              22200 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734914 ????????????????????????????????????????????????????????????
              22260 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9734974 ????????????????????????????????????????????????????????????
              22320 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735034 ????????????????????????????????????????????????????????????
              22380 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735094 ????????????????????????????????????????????????????????????
              22440 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735154 ????????????????????????????????????????????????????????????
              22500 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735214 ????????????????????????????????????????????????????????????
              22560 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735274 ????????????????????????????????????????????????????????????
              22620 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735334 ????????????????????????????????????????????????????????????
              22680 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735394 ????????????????????????????????????????????????????????????
              22740 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735454 ????????????????????????????????????????????????????????????
              22800 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735514 ????????????????????????????????????????????????????????????
              22860 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735574 ????????????????????????????????????????????????????????????
              22920 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735634 ????????????????????????????????????????????????????????????
              22980 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735694 ????????????????????????????????????????????????????????????
              23040 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735754 ????????????????????????????????????????????????????????????
              23100 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735814 ????????????????????????????????????????????????????????????
              23160 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735874 ????????????????????????????????????????????????????????????
              23220 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735934 ????????????????????????????????????????????????????????????
              23280 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9735994 ????????????????????????????????????????????????????????????
              23340 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736054 ????????????????????????????????????????????????????????????
              23400 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736114 ????????????????????????????????????????????????????????????
              23460 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736174 ????????????????????????????????????????????????????????????
              23520 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736234 ????????????????????????????????????????????????????????????
              23580 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736294 ????????????????????????????????????????????????????????????
              23640 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736354 ????????????????????????????????????????????????????????????
              23700 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736414 ????????????????????????????????????????????????????????????
              23760 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736474 ????????????????????????????????????????????????????????????
              23820 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736534 ????????????????????????????????????????????????????????????
              23880 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736594 ????????????????????????????????????????????????????????????
              23940 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736654 ????????????????????????????????????????????????????????????
              24000 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736714 ????????????????????????????????????????????????????????????
              24060 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736774 ????????????????????????????????????????????????????????????
              24120 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736834 ????????????????????????????????????????????????????????????
              24180 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736894 ????????????????????????????????????????????????????????????
              24240 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9736954 ????????????????????????????????????????????????????????????
              24300 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737014 ????????????????????????????????????????????????????????????
              24360 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737074 ????????????????????????????????????????????????????????????
              24420 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737134 ????????????????????????????????????????????????????????????
              24480 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737194 ????????????????????????????????????????????????????????????
              24540 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737254 ????????????????????????????????????????????????????????????
              24600 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737314 ????????????????????????????????????????????????????????????
              24660 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737374 ????????????????????????????????????????????????????????????
              24720 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737434 ????????????????????????????????????????????????????????????
              24780 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737494 ????????????????????????????????????????????????????????????
              24840 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737554 ????????????????????????????????????????????????????????????
              24900 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737614 ????????????????????????????????????????????????????????????
              24960 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737674 ????????????????????????????????????????????????????????????
              25020 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737734 ????????????????????????????????????????????????????????????
              25080 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737794 ????????????????????????????????????????????????????????????
              25140 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737854 ????????????????????????????????????????????????????????????
              25200 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737914 ????????????????????????????????????????????????????????????
              25260 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9737974 ????????????????????????????????????????????????????????????
              25320 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738034 ????????????????????????????????????????????????????????????
              25380 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738094 ????????????????????????????????????????????????????????????
              25440 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738154 ????????????????????????????????????????????????????????????
              25500 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738214 ????????????????????????????????????????????????????????????
              25560 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738274 ????????????????????????????????????????????????????????????
              25620 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738334 ????????????????????????????????????????????????????????????
              25680 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738394 ????????????????????????????????????????????????????????????
              25740 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738454 ????????????????????????????????????????????????????????????
              25800 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738514 ????????????????????????????????????????????????????????????
              25860 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738574 ????????????????????????????????????????????????????????????
              25920 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738634 ????????????????????????????????????????????????????????????
              25980 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738694 ????????????????????????????????????????????????????????????
              26040 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738754 ????????????????????????????????????????????????????????????
              26100 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738814 ????????????????????????????????????????????????????????????
              26160 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738874 ????????????????????????????????????????????????????????????
              26220 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738934 ????????????????????????????????????????????????????????????
              26280 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9738994 ????????????????????????????????????????????????????????????
              26340 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9739054 ????????????????????????????????????????????????????????????
              26400 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9739114 ????????????????????????????????????????????????????????????
              26460 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9739174 ????????????????????????????????????????????????????????????
              26520 ------------------------------------------------------------
CAG33136.       423 ------------------------------------------------------------

KI537979    9739234 ????????????????????????????????????????????????????????????
              26580 ------------------------------||||||||||||||||||||||||||||||
CAG33136.       423 ------------------------------??????????????????????????????

KI537979    9739294 ????????????????????????????????????????????????????????????
              26640 |||||||||||||||||||||||||||||||||||||||||||||---------------
CAG33136.       453 ?????????????????????????????????????????????---------------

KI537979    9739354 ????????????????????????????????????????????????????????????
              26700 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739414 ????????????????????????????????????????????????????????????
              26760 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739474 ????????????????????????????????????????????????????????????
              26820 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739534 ????????????????????????????????????????????????????????????
              26880 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739594 ????????????????????????????????????????????????????????????
              26940 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739654 ????????????????????????????????????????????????????????????
              27000 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739714 ????????????????????????????????????????????????????????????
              27060 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739774 ????????????????????????????????????????????????????????????
              27120 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739834 ????????????????????????????????????????????????????????????
              27180 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739894 ????????????????????????????????????????????????????????????
              27240 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9739954 ????????????????????????????????????????????????????????????
              27300 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740014 ????????????????????????????????????????????????????????????
              27360 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740074 ????????????????????????????????????????????????????????????
              27420 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740134 ????????????????????????????????????????????????????????????
              27480 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740194 ????????????????????????????????????????????????????????????
              27540 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740254 ????????????????????????????????????????????????????????????
              27600 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740314 ????????????????????????????????????????????????????????????
              27660 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740374 ????????????????????????????????????????????????????????????
              27720 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740434 ????????????????????????????????????????????????????????????
              27780 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740494 ????????????????????????????????????????????????????????????
              27840 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740554 ????????????????????????????????????????????????????????????
              27900 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740614 ????????????????????????????????????????????????????????????
              27960 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740674 ????????????????????????????????????????????????????????????
              28020 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740734 ????????????????????????????????????????????????????????????
              28080 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740794 ????????????????????????????????????????????????????????????
              28140 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740854 ????????????????????????????????????????????????????????????
              28200 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740914 ????????????????????????????????????????????????????????????
              28260 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9740974 ????????????????????????????????????????????????????????????
              28320 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741034 ????????????????????????????????????????????????????????????
              28380 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741094 ????????????????????????????????????????????????????????????
              28440 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741154 ????????????????????????????????????????????????????????????
              28500 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741214 ????????????????????????????????????????????????????????????
              28560 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741274 ????????????????????????????????????????????????????????????
              28620 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741334 ????????????????????????????????????????????????????????????
              28680 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741394 ????????????????????????????????????????????????????????????
              28740 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741454 ????????????????????????????????????????????????????????????
              28800 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741514 ????????????????????????????????????????????????????????????
              28860 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741574 ????????????????????????????????????????????????????????????
              28920 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741634 ????????????????????????????????????????????????????????????
              28980 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741694 ????????????????????????????????????????????????????????????
              29040 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741754 ????????????????????????????????????????????????????????????
              29100 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741814 ????????????????????????????????????????????????????????????
              29160 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741874 ????????????????????????????????????????????????????????????
              29220 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741934 ????????????????????????????????????????????????????????????
              29280 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9741994 ????????????????????????????????????????????????????????????
              29340 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742054 ????????????????????????????????????????????????????????????
              29400 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742114 ????????????????????????????????????????????????????????????
              29460 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742174 ????????????????????????????????????????????????????????????
              29520 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742234 ????????????????????????????????????????????????????????????
              29580 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742294 ????????????????????????????????????????????????????????????
              29640 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742354 ????????????????????????????????????????????????????????????
              29700 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742414 ????????????????????????????????????????????????????????????
              29760 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742474 ????????????????????????????????????????????????????????????
              29820 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742534 ????????????????????????????????????????????????????????????
              29880 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742594 ????????????????????????????????????????????????????????????
              29940 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742654 ????????????????????????????????????????????????????????????
              30000 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742714 ????????????????????????????????????????????????????????????
              30060 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742774 ????????????????????????????????????????????????????????????
              30120 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742834 ????????????????????????????????????????????????????????????
              30180 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742894 ????????????????????????????????????????????????????????????
              30240 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9742954 ????????????????????????????????????????????????????????????
              30300 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743014 ????????????????????????????????????????????????????????????
              30360 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743074 ????????????????????????????????????????????????????????????
              30420 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743134 ????????????????????????????????????????????????????????????
              30480 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743194 ????????????????????????????????????????????????????????????
              30540 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743254 ????????????????????????????????????????????????????????????
              30600 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743314 ????????????????????????????????????????????????????????????
              30660 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743374 ????????????????????????????????????????????????????????????
              30720 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743434 ????????????????????????????????????????????????????????????
              30780 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743494 ????????????????????????????????????????????????????????????
              30840 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743554 ????????????????????????????????????????????????????????????
              30900 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743614 ????????????????????????????????????????????????????????????
              30960 ------------------------------------------------------------
CAG33136.       498 ------------------------------------------------------------

KI537979    9743674 ????????????????????????????????????????????????????????????
              31020 --------------------------------||||||||||||||||||||||||||||
CAG33136.       498 --------------------------------????????????????????????????

KI537979    9743734 ????????????????????????????????????????????????????????????
              31080 ||||||||||||||||||||||||||||||||----------------------------
CAG33136.       526 ????????????????????????????????----------------------------

KI537979    9743794 ????????????????????????????????????????????????????????????
              31140 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9743854 ????????????????????????????????????????????????????????????
              31200 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9743914 ????????????????????????????????????????????????????????????
              31260 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9743974 ????????????????????????????????????????????????????????????
              31320 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9744034 ????????????????????????????????????????????????????????????
              31380 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9744094 ????????????????????????????????????????????????????????????
              31440 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9744154 ????????????????????????????????????????????????????????????
              31500 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9744214 ????????????????????????????????????????????????????????????
              31560 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9744274 ????????????????????????????????????????????????????????????
              31620 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9744334 ????????????????????????????????????????????????????????????
              31680 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9744394 ????????????????????????????????????????????????????????????
              31740 ------------------------------------------------------------
CAG33136.       558 ------------------------------------------------------------

KI537979    9744454 ????????????????????????????????????????????????????????????
              31800 ---------------------------------------------------------|||
CAG33136.       558 ---------------------------------------------------------???

KI537979    9744514 ????????????????????????????????????????????????????????????
              31860 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       561 ????????????????????????????????????????????????????????????

KI537979    9744574 ?????????????????? 9744592
              31920 ||||||||||||||||||   31938
CAG33136.       621 ??????????????????     639
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
KI537979	9712654	9744592	CAG33136.1	972	+	9712654	9744592	0	7	132,156,87,48,75,60,81,	0,3287,3791,5720,26610,31052,31857,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (639 aligned letters; 0 identities; 0 mismatches; 31299 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 639:
        identities = 0,
        mismatches = 0.
    gaps = 31299:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 31299:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 31299:
                open_internal_deletions = 6,
                extend_internal_deletions = 31293;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 31299)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 31299)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 31299)
        self.assertEqual(counts.gaps, 31299)
        self.assertEqual(counts.aligned, 639)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 792)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertLess(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "KI538594")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[2103463, 2103523, 2103522, 2104149],
                          [      0,      60,      60,     687]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
KI538594    2103463 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.         0 ????????????????????????????????????????????????????????????

KI538594    2103523 
                 60 
CAG33136.        60 

KI538594    2103522 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.        60 ????????????????????????????????????????????????????????????

KI538594    2103582 ????????????????????????????????????????????????????????????
                120 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       120 ????????????????????????????????????????????????????????????

KI538594    2103642 ????????????????????????????????????????????????????????????
                180 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       180 ????????????????????????????????????????????????????????????

KI538594    2103702 ????????????????????????????????????????????????????????????
                240 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       240 ????????????????????????????????????????????????????????????

KI538594    2103762 ????????????????????????????????????????????????????????????
                300 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       300 ????????????????????????????????????????????????????????????

KI538594    2103822 ????????????????????????????????????????????????????????????
                360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       360 ????????????????????????????????????????????????????????????

KI538594    2103882 ????????????????????????????????????????????????????????????
                420 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       420 ????????????????????????????????????????????????????????????

KI538594    2103942 ????????????????????????????????????????????????????????????
                480 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       480 ????????????????????????????????????????????????????????????

KI538594    2104002 ????????????????????????????????????????????????????????????
                540 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       540 ????????????????????????????????????????????????????????????

KI538594    2104062 ????????????????????????????????????????????????????????????
                600 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       600 ????????????????????????????????????????????????????????????

KI538594    2104122 ??????????????????????????? 2104149
                660 |||||||||||||||||||||||||||     687
CAG33136.       660 ???????????????????????????     687
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
KI538594	2103463	2104149	CAG33136.1	792	+	2103463	2104149	0	2	60,627,	0,59,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (687 aligned letters; 0 identities; 0 mismatches; -1 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 687:
        identities = 0,
        mismatches = 0.
    gaps = -1:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = -1:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = -1:
                open_internal_deletions = 1,
                extend_internal_deletions = -2;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, -1)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, -1)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, -1)
        self.assertEqual(counts.gaps, -1)
        self.assertEqual(counts.aligned, 687)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 902)
        self.assertLess(alignment.coordinates[0, 0], alignment.coordinates[0, -1])
        self.assertGreater(alignment.coordinates[1, 0], alignment.coordinates[1, -1])
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "KI537194")
        self.assertEqual(alignment.query.id, "CAG33136.1")
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[20872390, 20872471, 20872472, 20873021],
                          [     630,      549,      549,        0]]),
                # fmt: on
            )
        )
        self.assertEqual(
            str(alignment),
            """\
KI537194   20872390 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       630 ????????????????????????????????????????????????????????????

KI537194   20872450 ????????????????????????????????????????????????????????????
                 60 |||||||||||||||||||||-||||||||||||||||||||||||||||||||||||||
CAG33136.       570 ?????????????????????-??????????????????????????????????????

KI537194   20872510 ????????????????????????????????????????????????????????????
                120 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       511 ????????????????????????????????????????????????????????????

KI537194   20872570 ????????????????????????????????????????????????????????????
                180 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       451 ????????????????????????????????????????????????????????????

KI537194   20872630 ????????????????????????????????????????????????????????????
                240 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       391 ????????????????????????????????????????????????????????????

KI537194   20872690 ????????????????????????????????????????????????????????????
                300 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       331 ????????????????????????????????????????????????????????????

KI537194   20872750 ????????????????????????????????????????????????????????????
                360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       271 ????????????????????????????????????????????????????????????

KI537194   20872810 ????????????????????????????????????????????????????????????
                420 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       211 ????????????????????????????????????????????????????????????

KI537194   20872870 ????????????????????????????????????????????????????????????
                480 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.       151 ????????????????????????????????????????????????????????????

KI537194   20872930 ????????????????????????????????????????????????????????????
                540 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
CAG33136.        91 ????????????????????????????????????????????????????????????

KI537194   20872990 ??????????????????????????????? 20873021
                600 |||||||||||||||||||||||||||||||      631
CAG33136.        31 ???????????????????????????????        0
""",
        )
        self.assertEqual(
            format(alignment, "bed"),
            """\
KI537194	20872390	20873021	CAG33136.1	902	-	20872390	20873021	0	2	81,549,	0,82,
""",
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (630 aligned letters; 0 identities; 0 mismatches; 1 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 630:
        identities = 0,
        mismatches = 0.
    gaps = 1:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 1:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 1:
                open_internal_deletions = 1,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 1)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 1)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 1)
        self.assertEqual(counts.gaps, 1)
        self.assertEqual(counts.aligned, 630)
        self.assertRaises(StopIteration, next, alignments)

    def test_writing_psl_35_002(self):
        """Test writing the alignments in psl_35_002.bed."""
        path = "Blat/psl_35_002.bed"
        with open(path) as stream:
            original_data = stream.read()
        alignments = Align.parse(path, "bed")
        stream = StringIO()
        n = Align.write(alignments, stream, "bed")
        self.assertEqual(n, 3)
        stream.seek(0)
        written_data = stream.read()
        stream.close()
        self.assertEqual(original_data, written_data)


class TestAlign_bed12(unittest.TestCase):
    def test_reading(self):
        """Test parsing alignments in file formats BED3 through BED12."""
        for bedN in (3, 4, 5, 6, 7, 8, 9, 12):
            filename = "bed%d.bed" % bedN
            path = os.path.join("Blat", filename)
            alignments = Align.parse(path, "bed")
            alignment = next(alignments)
            if bedN >= 5:
                self.assertEqual(alignment.score, 960, msg=filename)
            self.assertEqual(alignment.shape, (2, 4000), msg=filename)
            self.assertLess(
                alignment.coordinates[0, 0], alignment.coordinates[0, -1], msg=filename
            )
            self.assertLess(
                alignment.coordinates[1, 0], alignment.coordinates[1, -1], msg=filename
            )
            self.assertEqual(len(alignment), 2, msg=filename)
            self.assertIs(alignment.sequences[0], alignment.target, msg=filename)
            self.assertIs(alignment.sequences[1], alignment.query, msg=filename)
            self.assertEqual(alignment.target.id, "chr22", msg=filename)
            if bedN >= 4:
                self.assertEqual(alignment.query.id, "mRNA1", msg=filename)
            else:
                self.assertIsNone(alignment.query.id, msg=filename)
            if bedN == 12:
                self.assertTrue(
                    np.array_equal(
                        alignment.coordinates,
                        # fmt: off
                        np.array([[1000, 1567, 4512, 5000],
                                  [   0,  567,  567, 1055]]),
                        # fmt: on
                    ),
                    msg=filename,
                )
            else:
                self.assertTrue(
                    np.array_equal(
                        alignment.coordinates,
                        np.array([[1000, 5000], [0, 4000]]),
                    ),
                    msg=filename,
                )
            if bedN >= 7:
                self.assertEqual(alignment.thickStart, 1200, msg=filename)
            if bedN >= 8:
                self.assertEqual(alignment.thickEnd, 4900, msg=filename)
            if bedN >= 9:
                self.assertEqual(alignment.itemRgb, "255,0,0", msg=filename)
            words = [
                "chr22",
                "1000",
                "5000",
                "mRNA1",
                "960",
                "+",
                "1200",
                "4900",
                "255,0,0",
                "2",
                "567,488,",
                "0,3512,",
            ]
            self.assertEqual(
                alignment.format("bed", bedN), "\t".join(words[:bedN]) + "\n"
            )
            counts = alignment.counts()
            if bedN <= 9:
                self.assertEqual(
                    repr(counts),
                    "<AlignmentCounts object (4000 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                    % id(counts),
                )
                self.assertEqual(
                    str(counts),
                    """\
AlignmentCounts object with
    aligned = 4000:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
                )
            else:  # bedN == 12
                self.assertEqual(
                    repr(counts),
                    "<AlignmentCounts object (1055 aligned letters; 0 identities; 0 mismatches; 2945 gaps) at 0x%x>"
                    % id(counts),
                )
                self.assertEqual(
                    str(counts),
                    """\
AlignmentCounts object with
    aligned = 1055:
        identities = 0,
        mismatches = 0.
    gaps = 2945:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 2945:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 2945:
                open_internal_deletions = 1,
                extend_internal_deletions = 2944;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
                )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            if bedN == 12:
                self.assertEqual(counts.internal_deletions, 2945)
                self.assertEqual(counts.internal_gaps, 2945)
                self.assertEqual(counts.deletions, 2945)
                self.assertEqual(counts.gaps, 2945)
                self.assertEqual(counts.aligned, 1055)
            else:
                self.assertEqual(counts.internal_deletions, 0)
                self.assertEqual(counts.internal_gaps, 0)
                self.assertEqual(counts.deletions, 0)
                self.assertEqual(counts.gaps, 0)
                self.assertEqual(counts.aligned, 4000)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            alignment = next(alignments)
            if bedN >= 5:
                self.assertEqual(alignment.score, 900, msg=filename)
            self.assertEqual(alignment.shape, (2, 4000), msg=filename)
            self.assertLess(
                alignment.coordinates[0, 0], alignment.coordinates[0, -1], msg=filename
            )
            if bedN >= 6:
                self.assertGreater(
                    alignment.coordinates[1, 0],
                    alignment.coordinates[1, -1],
                    msg=filename,
                )
            else:
                self.assertLess(
                    alignment.coordinates[1, 0],
                    alignment.coordinates[1, -1],
                    msg=filename,
                )
            self.assertEqual(len(alignment), 2, msg=filename)
            self.assertIs(alignment.sequences[0], alignment.target, msg=filename)
            self.assertIs(alignment.sequences[1], alignment.query, msg=filename)
            self.assertEqual(alignment.target.id, "chr22", msg=filename)
            if bedN >= 4:
                self.assertEqual(alignment.query.id, "mRNA2", msg=filename)
            else:
                self.assertIsNone(alignment.query.id, msg=filename)
            if bedN == 12:
                self.assertTrue(
                    np.array_equal(
                        alignment.coordinates,
                        # fmt: off
                        np.array([[2000, 2433, 5601, 6000],
                                  [ 832,  399,  399,    0]])
                        # fmt: on
                    ),
                    msg=filename,
                )
            elif bedN >= 6:
                self.assertTrue(
                    np.array_equal(
                        alignment.coordinates,
                        np.array([[2000, 6000], [4000, 0]]),
                    ),
                    msg=filename,
                )
            else:
                self.assertTrue(
                    np.array_equal(
                        alignment.coordinates,
                        np.array([[2000, 6000], [0, 4000]]),
                    ),
                    msg=filename,
                )
            if bedN >= 7:
                self.assertEqual(alignment.thickStart, 2300, msg=filename)
            if bedN >= 8:
                self.assertEqual(alignment.thickEnd, 5960, msg=filename)
            if bedN >= 9:
                self.assertEqual(alignment.itemRgb, "0,255,0", msg=filename)
            words = [
                "chr22",
                "2000",
                "6000",
                "mRNA2",
                "900",
                "-",
                "2300",
                "5960",
                "0,255,0",
                "2",
                "433,399,",
                "0,3601,",
            ]
            self.assertEqual(
                alignment.format("bed", bedN), "\t".join(words[:bedN]) + "\n"
            )
            counts = alignment.counts()
            if bedN <= 9:
                self.assertEqual(
                    repr(counts),
                    "<AlignmentCounts object (4000 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                    % id(counts),
                )
                self.assertEqual(
                    str(counts),
                    """\
AlignmentCounts object with
    aligned = 4000:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
                )
            else:  # bedN == 12
                self.assertEqual(
                    repr(counts),
                    "<AlignmentCounts object (832 aligned letters; 0 identities; 0 mismatches; 3168 gaps) at 0x%x>"
                    % id(counts),
                )
                self.assertEqual(
                    str(counts),
                    """\
AlignmentCounts object with
    aligned = 832:
        identities = 0,
        mismatches = 0.
    gaps = 3168:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 3168:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 3168:
                open_internal_deletions = 1,
                extend_internal_deletions = 3167;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
                )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            if bedN == 12:
                self.assertEqual(counts.internal_deletions, 3168)
                self.assertEqual(counts.deletions, 3168)
                self.assertEqual(counts.internal_gaps, 3168)
                self.assertEqual(counts.gaps, 3168)
                self.assertEqual(counts.aligned, 832)
            else:
                self.assertEqual(counts.internal_deletions, 0)
                self.assertEqual(counts.deletions, 0)
                self.assertEqual(counts.internal_gaps, 0)
                self.assertEqual(counts.gaps, 0)
                self.assertEqual(counts.aligned, 4000)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            with self.assertRaises(StopIteration) as cm:
                next(alignments)
                self.fail(f"More than two alignments reported in {filename}")

    def test_writing(self):
        """Test writing the alignments in bed12.bed as BED3 through BED12."""
        for bedN in (3, 4, 5, 6, 7, 8, 9, 12):
            filename = "bed%d.bed" % bedN
            path = os.path.join("Blat", filename)
            with open(path) as stream:
                original_data = stream.read()
            alignments = Align.parse(path, "bed")
            stream = StringIO()
            n = Align.write(alignments, stream, "bed", bedN=bedN)
            self.assertEqual(n, 2, msg=filename)
            stream.seek(0)
            written_data = stream.read()
            stream.close()
            self.assertEqual(original_data, written_data, msg=filename)


class TestAlign_strand(unittest.TestCase):
    def test_format(self):
        """Test alignment with the target on the opposite strand."""
        sequences = ["AACAGCAGCGTGTCG", "CAGCTAGCGAA"]
        coordinates = np.array(
            [[0, 2, 2, 3, 4, 6, 6, 9, 10, 12, 15], [11, 11, 9, 8, 8, 6, 5, 2, 2, 0, 0]]
        )
        alignment = Alignment(sequences, coordinates)
        alignment.score = 8
        line = """\
target	2	12	query	8	-	2	12	0	4	1,2,3,2,	0,2,4,8,
"""
        self.assertEqual(
            str(alignment),
            """\
target            0 AA--CAGC-AGCGTGTCG 15
                  0 ----|-||-|||-||--- 18
query            11 --TTC-GCTAGC-TG---  0
""",
        )
        self.assertEqual(format(alignment, "bed"), line)
        alignment.coordinates = alignment.coordinates[:, ::-1]
        self.assertEqual(
            str(alignment),
            """\
target           15 CGACACGCT-GCTG--TT  0
                  0 ---||-|||-||-|---- 18
query             0 ---CA-GCTAGC-GAA-- 11
""",
        )
        self.assertEqual(format(alignment, "bed"), line)
        alignment.coordinates = alignment.coordinates[:, ::-1]
        line = """\
target	3	13	query	0	-	3	13	0	4	2,3,2,1,	0,3,6,9,
"""
        alignment = alignment.reverse_complement()
        self.assertEqual(
            str(alignment),
            """\
target            0 CGACACGCT-GCTG--TT 15
                  0 ---||-|||-||-|---- 18
query            11 ---CA-GCTAGC-GAA--  0
""",
        )
        self.assertEqual(format(alignment, "bed"), line)
        alignment.coordinates = alignment.coordinates[:, ::-1]
        self.assertEqual(
            str(alignment),
            """\
target           15 AA--CAGC-AGCGTGTCG  0
                  0 ----|-||-|||-||--- 18
query             0 --TTC-GCTAGC-TG--- 11
""",
        )
        self.assertEqual(format(alignment, "bed"), line)


class TestAlign_searching(unittest.TestCase):
    path = "Blat/bigbedtest.bed"

    def check_alignments(self, alignments):
        alignment = next(alignments)
        self.assertEqual(alignment.score, 1)
        self.assertEqual(alignment.shape, (2, 90))
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "name1")
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[10, 100], [0, 90]]))
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (90 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 90:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 90)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 2)
        self.assertEqual(alignment.shape, (2, 10))
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "name2")
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[29, 39], [10, 0]]))
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (10 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 10:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 10)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 3)
        self.assertEqual(alignment.shape, (2, 100))
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr1")
        self.assertEqual(alignment.query.id, "name3")
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[200, 300], [0, 100]]))
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (100 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 100:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 100)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 6)
        self.assertEqual(alignment.shape, (2, 0))
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "name4")
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[50, 50], [0, 0]]))
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 0)
        self.assertEqual(counts.identities, 0)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 4)
        self.assertEqual(alignment.shape, (2, 10))
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "name5")
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[100, 110], [0, 10]]))
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (10 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 10:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 10)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 5)
        self.assertEqual(alignment.shape, (2, 10))
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "name6")
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[200, 210], [0, 10]]))
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (10 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 10:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 10)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 6)
        self.assertEqual(alignment.shape, (2, 0))
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr2")
        self.assertEqual(alignment.query.id, "name7")
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[220, 220], [0, 0]]))
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 0)
        self.assertEqual(counts.identities, 0)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(alignment.score, 7)
        self.assertEqual(alignment.shape, (2, 0))
        self.assertEqual(len(alignment), 2)
        self.assertIs(alignment.sequences[0], alignment.target)
        self.assertIs(alignment.sequences[1], alignment.query)
        self.assertEqual(alignment.target.id, "chr3")
        self.assertEqual(alignment.query.id, "name8")
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[0, 0], [0, 0]]))
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 0)
        self.assertEqual(counts.identities, 0)
        self.assertEqual(counts.mismatches, 0)

    def test_reading(self):
        """Test reading bigbedtest.bed."""
        alignments = Align.parse(self.path, "bed")
        self.check_alignments(alignments)

    def test_writing(self):
        """Test writing bigbedtest.bed."""
        alignments = Align.parse(self.path, "bed")
        with tempfile.TemporaryFile("w+t") as output:
            Align.write(alignments, output, "bed", bedN=6)
            output.seek(0)
            alignments = Align.parse(output, "bed")
            self.check_alignments(alignments)


if __name__ == "__main__":
    runner = unittest.TextTestRunner(verbosity=2)
    unittest.main(testRunner=runner)
