<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\TypeInfo\Tests\TypeResolver;

use PHPUnit\Framework\TestCase;
use Symfony\Component\TypeInfo\Exception\UnsupportedException;
use Symfony\Component\TypeInfo\Tests\Fixtures\Dummy;
use Symfony\Component\TypeInfo\Type;
use Symfony\Component\TypeInfo\TypeResolver\TypeResolver;
use Symfony\Component\TypeInfo\TypeResolver\TypeResolverInterface;

class TypeResolverTest extends TestCase
{
    public function testResolve()
    {
        $resolver = TypeResolver::create();

        $this->assertEquals(Type::bool(), $resolver->resolve('bool'));
        $this->assertEquals(Type::int(), $resolver->resolve((new \ReflectionProperty(Dummy::class, 'id'))->getType()));
        $this->assertEquals(Type::int(), $resolver->resolve((new \ReflectionMethod(Dummy::class, 'setId'))->getParameters()[0]));
        $this->assertEquals(Type::int(), $resolver->resolve(new \ReflectionProperty(Dummy::class, 'id')));
        $this->assertEquals(Type::void(), $resolver->resolve(new \ReflectionMethod(Dummy::class, 'setId')));
        $this->assertEquals(Type::string(), $resolver->resolve(new \ReflectionFunction(strtoupper(...))));
    }

    public function testCannotFindResolver()
    {
        $this->expectException(UnsupportedException::class);
        $this->expectExceptionMessage('Cannot find any resolver for "int" type.');

        $resolver = TypeResolver::create([]);
        $resolver->resolve(1);
    }

    public function testUseProperResolver()
    {
        $stringResolver = $this->createStub(TypeResolverInterface::class);
        $stringResolver->method('resolve')->willReturn(Type::template('STRING'));

        $reflectionTypeResolver = $this->createStub(TypeResolverInterface::class);
        $reflectionTypeResolver->method('resolve')->willReturn(Type::template('REFLECTION_TYPE'));

        $reflectionParameterResolver = $this->createStub(TypeResolverInterface::class);
        $reflectionParameterResolver->method('resolve')->willReturn(Type::template('REFLECTION_PARAMETER'));

        $reflectionPropertyResolver = $this->createStub(TypeResolverInterface::class);
        $reflectionPropertyResolver->method('resolve')->willReturn(Type::template('REFLECTION_PROPERTY'));

        $reflectionReturnTypeResolver = $this->createStub(TypeResolverInterface::class);
        $reflectionReturnTypeResolver->method('resolve')->willReturn(Type::template('REFLECTION_RETURN_TYPE'));

        $resolver = TypeResolver::create([
            'string' => $stringResolver,
            \ReflectionType::class => $reflectionTypeResolver,
            \ReflectionParameter::class => $reflectionParameterResolver,
            \ReflectionProperty::class => $reflectionPropertyResolver,
            \ReflectionFunctionAbstract::class => $reflectionReturnTypeResolver,
        ]);

        $this->assertEquals(Type::template('STRING'), $resolver->resolve('foo'));
        $this->assertEquals(
            Type::template('REFLECTION_TYPE'),
            $resolver->resolve((new \ReflectionProperty(Dummy::class, 'id'))->getType()),
        );
        $this->assertEquals(
            Type::template('REFLECTION_PARAMETER'),
            $resolver->resolve((new \ReflectionMethod(Dummy::class, 'setId'))->getParameters()[0]),
        );
        $this->assertEquals(
            Type::template('REFLECTION_PROPERTY'),
            $resolver->resolve(new \ReflectionProperty(Dummy::class, 'id')),
        );
        $this->assertEquals(
            Type::template('REFLECTION_RETURN_TYPE'),
            $resolver->resolve(new \ReflectionMethod(Dummy::class, 'setId')),
        );
        $this->assertEquals(
            Type::template('REFLECTION_RETURN_TYPE'),
            $resolver->resolve(new \ReflectionFunction(strtoupper(...))),
        );
    }
}
